#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>

#include "graph.h"
#include "utils.h"

#define INF UINT_MAX

typedef unsigned (*cost_fn)(Route *);

static void dijkstra(const Graph *g, unsigned src,
		     unsigned *dist, int *prev,
		     cost_fn cost)
{
	unsigned n = g->n_nodes;
	int *visited = calloc(n, sizeof(int));
	DIE(!visited, "calloc failed\n");

	for (unsigned i = 0; i < n; ++i) {
		dist[i] = INF;
		prev[i] = -1;
	}

	dist[src] = 0;

	for (unsigned step = 0; step < n; ++step) {
		/* pick unvisited node with smallest distance */
		unsigned u = INF;
		for (unsigned i = 0; i < n; ++i)
			if (!visited[i] && dist[i] != INF &&
			    (u == INF || dist[i] < dist[u]))
				u = i;

		if (u == INF)
			break;

		visited[u] = 1;

		for (Route *r = g->routes[u]; r; r = r->next) {
			unsigned v = r->dst->id;
			unsigned w = cost(r);

			if (!visited[v] && dist[u] + w < dist[v]) {
				dist[v] = dist[u] + w;
				prev[v] = (int)u;
			}
		}
	}

	free(visited);
}



static void print_connected_components(const Graph *g,
				       const char *template)
{
	unsigned n = g->n_nodes;
	int *visited = calloc(n, sizeof(int));
	DIE(!visited, "calloc failed\n");

	char filename[512];
	snprintf(filename, sizeof(filename), "%s-connected.txt", template);

	FILE *out = fopen(filename, "w");
	DIE(!out, "Could not open %s\n", filename);

	/* adjacency check ignoring direction */
	for (unsigned start = 0; start < n; ++start) {
		if (visited[start])
			continue;

		/* BFS queue */
		unsigned *queue = malloc(n * sizeof(unsigned));
		DIE(!queue, "malloc failed\n");

		unsigned qh = 0, qt = 0;
		queue[qt++] = start;
		visited[start] = 1;

		while (qh < qt) {
			unsigned u = queue[qh++];

			fprintf(out, "%s ", g->airports[u].iata);

			/* outgoing edges */
			for (Route *r = g->routes[u]; r; r = r->next) {
				unsigned v = r->dst->id;
				if (!visited[v]) {
					visited[v] = 1;
					queue[qt++] = v;
				}
			}

			/* incoming edges (scan graph) */
			for (unsigned v = 0; v < n; ++v) {
				for (Route *r = g->routes[v]; r; r = r->next) {
					if (r->dst->id == u && !visited[v]) {
						visited[v] = 1;
						queue[qt++] = v;
					}
				}
			}
		}

		fprintf(out, "\n");
		free(queue);
	}

	fclose(out);
	free(visited);
}



static void print_path(FILE *out, const Graph *g,
		       int *prev, unsigned src, unsigned v)
{
	if (v == src) {
		fprintf(out, "%s", g->airports[v].iata);
		return;
	}
	if (prev[v] == -1)
		return;

	print_path(out, g, prev, src, prev[v]);
	fprintf(out, " -> %s", g->airports[v].iata);
}

static void write_output(const char *filename,
			 const Graph *g,
			 unsigned src,
			 unsigned *dist,
			 int *prev)
{
	FILE *out = fopen(filename, "w");
	DIE(!out, "Could not open %s\n", filename);

	for (unsigned i = 0; i < g->n_nodes; ++i) {
		fprintf(out, "%s: ", g->airports[i].iata);

		if (i == src) {
			fprintf(out, "-\n");
			continue;
		}

		if (dist[i] == INF) {
			fprintf(out, "unreachable\n");
			continue;
		}

		fprintf(out, "%u (", dist[i]);
		print_path(out, g, prev, src, i);
		fprintf(out, ")\n");
	}

	fclose(out);
}

int main(int argc, char **argv)
{
	if (argc != 4) {
		fprintf(stderr,
			"Usage: %s <routes_file> <IATA> <output_template>\n",
			argv[0]);
		return EXIT_FAILURE;
	}

	const char *path = argv[1];
	const char *iata = argv[2];
	const char *tpl  = argv[3];

	Graph *graph = load_graph_from_file(path);
	DIE(!graph, "Error: could not load graph from '%s'\n", path);

	/* find source airport */
	unsigned src = INF;
	for (unsigned i = 0; i < graph->n_nodes; ++i)
		if (strcmp(graph->airports[i].iata, iata) == 0)
			src = i;

	DIE(src == INF, "Airport %s not found\n", iata);

	unsigned *dist = malloc(graph->n_nodes * sizeof(unsigned));
	int *prev = malloc(graph->n_nodes * sizeof(int));
	DIE(!dist || !prev, "malloc failed\n");

	char fname[4096];

	/* distance (km) */
	dijkstra(graph, src, dist, prev, cost_km);
	snprintf(fname, sizeof(fname), "%s-distance.txt", tpl);
	write_output(fname, graph, src, dist, prev);

	/* time (minutes) */
	dijkstra(graph, src, dist, prev, cost_min);
	snprintf(fname, sizeof(fname), "%s-time.txt", tpl);
	write_output(fname, graph, src, dist, prev);

	/* hops*/
	dijkstra(graph, src, dist, prev, cost_hops);
	snprintf(fname, sizeof(fname), "%s-hops.txt", tpl);
	write_output(fname, graph, src, dist, prev);

	print_connected_components(graph, tpl);

	free(dist);
	free(prev);
	free_graph(graph);
	return EXIT_SUCCESS;
}
