#include "graph.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// FIXME: it would be much better to allow lines of arbitrary length.
#define MAX_LINE_LEN 64000

Graph *allocate_graph(unsigned n_nodes, unsigned n_edges) {
	Graph *g = malloc(sizeof(Graph));
	if (!g)
		return NULL;

	g->n_nodes = n_nodes;
	g->n_edges = n_edges;

	g->airports = calloc(n_nodes, sizeof(Airport));
	g->routes = calloc(n_nodes, sizeof(Route *));
	if (!g->airports || !g->routes) {
		free(g->airports);
		free(g->routes);
		free(g);
		return NULL;
	}

	return g;
}

void add_route(Graph *g, unsigned src_id, unsigned dst_id, unsigned cost_km,
	       unsigned cost_min) {
	Route *r = malloc(sizeof(Route));
	if (!r)
		return;

	r->dst = &g->airports[dst_id];
	r->cost_km = cost_km;
	r->cost_min = cost_min;
	r->next = g->routes[src_id];

	g->routes[src_id] = r;
}

Graph *load_graph_from_file(const char *path) {
	FILE *f = fopen(path, "r");
	if (!f)
		return NULL;

	unsigned n_nodes, n_edges;
	if (fscanf(f, "%u %u\n", &n_nodes, &n_edges) != 2) {
		fclose(f);
		return NULL;
	}

	Graph *g = allocate_graph(n_nodes, n_edges);
	if (!g) {
		fclose(f);
		return NULL;
	}

	char line[MAX_LINE_LEN];
	for (unsigned i = 0; i < n_nodes; ++i) {
		if (!fgets(line, sizeof(line), f)) {
			free_graph(g);
			fclose(f);
			return NULL;
		}

		char *token = strtok(line, " \n");
		if (!token)
			continue;

		strncpy(g->airports[i].iata, token, IATA_LENGTH - 1);
		g->airports[i].iata[IATA_LENGTH - 1] = '\0';
		g->airports[i].id = i;

		while (1) {
			char *dst_tok = strtok(NULL, " \n");
			char *km_tok = strtok(NULL, " \n");
			char *min_tok = strtok(NULL, " \n");

			if (!dst_tok || !km_tok || !min_tok)
				break;

			unsigned dst_id = (unsigned)atoi(dst_tok);
			unsigned cost_km = (unsigned)atoi(km_tok);
			unsigned cost_min = (unsigned)atoi(min_tok);

			if (dst_id < n_nodes) {
				add_route(g, i, dst_id, cost_km, cost_min);
			}
		}
	}

	fclose(f);
	return g;
}

void free_graph(Graph *g) {
	if (!g)
		return;

	for (unsigned i = 0; i < g->n_nodes; ++i) {
		Route *r = g->routes[i];
		while (r) {
			Route *next = r->next;
			free(r);
			r = next;
		}
	}

	free(g->routes);
	free(g->airports);
	free(g);
}
