#include "graph.h"

#include <stdio.h>
#include <stdlib.h>

int main(int argc, char **argv)
{
	if (argc != 4) {
		fprintf(stderr,
			"Usage: %s <input_path> <K> <output_path>\n",
			argv[0]);
		return EXIT_FAILURE;
	}

	const char *input_path  = argv[1];
	unsigned K = (unsigned)atoi(argv[2]);
	const char *output_path = argv[3];

	Graph *g = load_graph_from_file(input_path);
	if (!g) {
		fprintf(stderr, "Error: failed to load graph from '%s'\n", input_path);
		return EXIT_FAILURE;
	}

	unsigned *cover = improved_vertex_cover(g, K);

	if (cover) {
		printf("Vertex cover of size %u found:\n", K);
		for (unsigned i = 0; i < K; ++i) {
			printf("%u ", cover[i]);
		}
		printf("\n");
	} else {
		printf("No vertex cover of size %u exists.\n", K);
	}

	char *dot = graph_to_graphviz(g);
	if (!dot) {
		fprintf(stderr, "Error: failed to generate Graphviz output\n");
		free(cover);
		free_graph(g);
		return EXIT_FAILURE;
	}

	FILE *out = fopen(output_path, "w");
	if (!out) {
		fprintf(stderr, "Error: failed to open output file '%s'\n", output_path);
		free(dot);
		free(cover);
		free_graph(g);
		return EXIT_FAILURE;
	}

	fputs(dot, out);
	fclose(out);

	free(dot);
	free(cover);
	free_graph(g);

	return EXIT_SUCCESS;
}
