#ifndef GRAPH_H
#define GRAPH_H

#include <stddef.h>

typedef enum {
	GRAPH_DIRECTED,
	GRAPH_UNDIRECTED
} GraphType;

typedef struct Route {
	unsigned dst_id;
	unsigned cost;
	struct Route *next;
} Route;

typedef struct {
	unsigned n_nodes;
	unsigned n_edges;
	GraphType type;
	Route **routes;   /* adjacency lists, indexed by src node id */
} Graph;

Graph *allocate_graph(unsigned n_nodes, unsigned n_edges, GraphType type);

Graph *load_graph_from_file(const char *path);

void add_edge(Graph *g, unsigned src, unsigned dst, unsigned cost);

void free_graph(Graph *graph);

static unsigned route_cost(Route *r) { return r->cost; }
static unsigned cost_hops(Route *r) { (void)r; return 1; }

char *graph_to_graphviz(const Graph *g);

/* Returns an array of node ids of size K if a vertex cover exists,
 * otherwise returns NULL.
 * Caller owns the returned array.
 */
unsigned *vertex_cover(const Graph *g, unsigned K);

/* Checks whether `cover` (size k) is a valid vertex cover */
int is_vertex_cover(const Graph *g, unsigned k,
		    const unsigned *cover, const unsigned cover_size);

/* Uses kernelization */
unsigned *improved_vertex_cover(const Graph *g, unsigned K);

#endif /* GRAPH_H */
