#include "graph.h"
#include "utils.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>

// FIXME: it would be much better to allow lines of arbitrary length.
#define MAX_LINE_LEN 64000

Graph *allocate_graph(unsigned n_nodes, unsigned n_edges, GraphType type)
{
	Graph *g = malloc(sizeof(Graph));
	DIE(!g, "Failed to allocate graph\n");

	g->n_nodes = n_nodes;
	g->n_edges = n_edges;
	g->type = type;

	g->routes = calloc(n_nodes, sizeof(Route *));
	if (!g->routes) {
		free(g);
		return NULL;
	}

	return g;
}

void add_directed_edge(Graph *g, unsigned src, unsigned dst,
		       unsigned cost)
{
	Route *r = malloc(sizeof(Route));
	DIE(!r, "Failed to allocate route\n");

	r->dst_id = dst;
	r->cost = cost;
	r->next = g->routes[src];
	g->routes[src] = r;
}

void add_edge(Graph *g, unsigned src, unsigned dst, unsigned cost) {
	if (!g || src >= g->n_nodes || dst >= g->n_nodes)
		return;

	add_directed_edge(g, src, dst, cost);

	if (g->type == GRAPH_UNDIRECTED) {
		add_directed_edge(g, dst, src, cost);
	}
}


Graph *load_graph_from_file(const char *path)
{
	FILE *f = fopen(path, "r");
	if (!f)
		return NULL;

	char type_char;
	unsigned n_nodes, n_edges;

	if (fscanf(f, " %c %u %u\n", &type_char, &n_nodes, &n_edges) != 3) {
		fclose(f);
		return NULL;
	}

	GraphType type;
	if (type_char == 'D')
		type = GRAPH_DIRECTED;
	else if (type_char == 'U')
		type = GRAPH_UNDIRECTED;
	else {
		fclose(f);
		return NULL;
	}

	Graph *g = allocate_graph(n_nodes, n_edges, type);
	if (!g) {
		fclose(f);
		return NULL;
	}

	unsigned u, v, cost;
	while (fscanf(f, "%u %u %u", &u, &v, &cost) == 3) {
		add_edge(g, u, v, cost);
	}

	fclose(f);
	return g;
}

void free_graph(Graph *g) {
	if (!g)
		return;

	for (unsigned i = 0; i < g->n_nodes; ++i) {
		Route *r = g->routes[i];
		while (r) {
			Route *next = r->next;
			free(r);
			r = next;
		}
	}

	free(g->routes);
	free(g);
}


/* simple dynamic string builder */
static int appendf(char **buf, size_t *cap, size_t *len,
		   const char *fmt, ...) {
	while (1) {
		if (*len + 128 >= *cap) {
			size_t new_cap = (*cap == 0) ? 1024 : (*cap * 2);
			char *tmp = realloc(*buf, new_cap);
			if (!tmp)
				return 0;
			*buf = tmp;
			*cap = new_cap;
		}

		va_list ap;
		va_start(ap, fmt);
		int written = vsnprintf(*buf + *len, *cap - *len, fmt, ap);
		va_end(ap);

		if (written < 0)
			return 0;

		if ((size_t)written < *cap - *len) {
			*len += (size_t)written;
			return 1;
		}

		/* not enough space, grow and retry */
	}
}

char *graph_to_graphviz(const Graph *g) {
	if (!g)
		return NULL;

	char *buf = NULL;
	size_t cap = 0, len = 0;

	const char *graph_kw = (g->type == GRAPH_DIRECTED) ? "digraph" : "graph";
	const char *edge_op  = (g->type == GRAPH_DIRECTED) ? "->"      : "--";

	appendf(&buf, &cap, &len, "%s G {\n", graph_kw);

	for (unsigned u = 0; u < g->n_nodes; ++u) {
		for (Route *r = g->routes[u]; r; r = r->next) {

			/* avoid duplicates in undirected graphs */
			if (g->type == GRAPH_UNDIRECTED && u > r->dst_id)
				continue;

			appendf(&buf, &cap, &len,
				"    %u %s %u [label=\"%u\"];\n",
				u, edge_op, r->dst_id, r->cost);
		}
	}

	appendf(&buf, &cap, &len, "}\n");
	return buf;
}

static int contains(const unsigned *arr, unsigned n, unsigned x)
{
	for (unsigned i = 0; i < n; ++i)
		if (arr[i] == x)
			return 1;
	return 0;
}

int is_k_vertex_cover(const Graph *g, unsigned k,
		    const unsigned *cover, const unsigned cover_size)
{
	if (!g || !cover)
		return 0;

	if (cover_size < k)
		return 0;

	for (unsigned u = 0; u < g->n_nodes; ++u) {
		for (Route *r = g->routes[u]; r; r = r->next) {
			unsigned v = r->dst_id;

			/* Treat edges as undirected */
			if (!contains(cover, k, u) &&
			    !contains(cover, k, v)) {
				return 0;
			}
		}
	}
	return 1;
}

static int vc_backtrack(const Graph *g,
			unsigned K,
			unsigned start,
			unsigned depth,
			unsigned *solution)
{
	int isKCover = is_k_vertex_cover(g, K, solution, depth);
	if (isKCover)
		return 1;

	if (start >= g->n_nodes)
		return 0;

	for (unsigned v = start; v < g->n_nodes; ++v) {
		solution[depth] = v;

		if (vc_backtrack(g, K, v + 1, depth + 1, solution))
			return 1;
	}

	return 0;
}

unsigned *vertex_cover(const Graph *g, unsigned K)
{
	if (!g || K > g->n_nodes)
		return NULL;

	unsigned *solution = malloc(K * sizeof(unsigned));
	if (!solution)
		return NULL;

	if (vc_backtrack(g, K, 0, 0, solution)) {
		return solution;
	}

	free(solution);
	return NULL;
}

static unsigned degree(const Graph *g, unsigned v) {
	unsigned d = 0;
	for (Route *r = g->routes[v]; r; r = r->next)
		++d;
	return d;
}

static unsigned edge_count(const Graph *g) {
	unsigned cnt = 0;
	for (unsigned u = 0; u < g->n_nodes; ++u)
		for (Route *r = g->routes[u]; r; r = r->next)
			++cnt;

	return (g->type == GRAPH_UNDIRECTED) ? cnt / 2 : cnt;
}

static void remove_vertex(unsigned v, int *alive) {
	alive[v] = 0;
}

static int is_vertex_cover_masked(const Graph *g,
				  const unsigned *cover,
				  unsigned k,
				  const int *alive) {
	for (unsigned u = 0; u < g->n_nodes; ++u) {
		if (!alive[u])
			continue;

		for (Route *r = g->routes[u]; r; r = r->next) {
			unsigned v = r->dst_id;
			if (!alive[v])
				continue;

			int covered = 0;
			for (unsigned i = 0; i < k; ++i)
				if (cover[i] == u || cover[i] == v)
					covered = 1;

			if (!covered)
				return 0;
		}
	}
	return 1;
}

unsigned *improved_vertex_cover(const Graph *g, unsigned K) {
	if (!g)
		return NULL;

	int *alive = calloc(g->n_nodes, sizeof(int));
	if (!alive)
		return NULL;

	for (unsigned i = 0; i < g->n_nodes; ++i)
		alive[i] = 1;

	unsigned *forced = malloc(g->n_nodes * sizeof(unsigned));
	unsigned forced_cnt = 0;

	unsigned k = K;

	int changed;
	do {
		changed = 0;

		/* Rule 1: high-degree vertices */
		for (unsigned v = 0; v < g->n_nodes && k > 0; ++v) {
			if (!alive[v])
				continue;

			if (degree(g, v) > k) {
				forced[forced_cnt++] = v;
				remove_vertex(v, alive);
				--k;
				changed = 1;
			}
		}

		/* Rule 2: isolated vertices */
		for (unsigned v = 0; v < g->n_nodes; ++v) {
			if (!alive[v])
				continue;

			if (degree(g, v) == 0) {
				remove_vertex(v, alive);
				changed = 1;
			}
		}

		/* Rule 3: edge bound */
		if (!changed) {
			unsigned m = edge_count(g);
			if (m > k * k) {
				free(alive);
				free(forced);
				return NULL;
			}
		}

	} while (changed);

	/* Build reduced vertex list */
	unsigned *map = malloc(g->n_nodes * sizeof(unsigned));
	unsigned inv_cnt = 0;
	for (unsigned i = 0; i < g->n_nodes; ++i)
		if (alive[i])
			map[inv_cnt++] = i;

	/* Call naive solver on reduced graph */
	unsigned *sub_cover = vertex_cover(g, k);
	if (!sub_cover) {
		free(alive);
		free(forced);
		free(map);
		return NULL;
	}

	/* Reconstruct full solution */
	unsigned *result = malloc(K * sizeof(unsigned));
	unsigned idx = 0;

	for (unsigned i = 0; i < forced_cnt; ++i)
		result[idx++] = forced[i];

	for (unsigned i = 0; i < k; ++i)
		result[idx++] = sub_cover[i];

	free(sub_cover);
	free(map);
	free(forced);
	free(alive);

	return result;
}
