#!/usr/bin/env python3
import argparse
from graphviz import Digraph


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="Render airport graph into a PNG using Graphviz."
    )
    parser.add_argument(
        "input",
        help="Input graph file",
    )
    parser.add_argument(
        "--output",
        default="graph",
        help="Output file name without extension (default: graph)",
    )
    return parser.parse_args()


def main() -> int:
    args = parse_args()

    with open(args.input, "r", encoding="utf-8") as fin:
        lines = [line.strip() for line in fin if line.strip()]

    # First line: number of nodes and edges (not strictly needed)
    header = lines[0]
    graph_lines = lines[1:]

    dot = Digraph(
        format="png",
        graph_attr={"rankdir": "LR"},
        node_attr={
            "shape": "circle",
            "fixedsize": "true",
            "width": "0.9",
        },
        edge_attr={"fontsize": "10"},
    )

    # First pass: add nodes
    nodes = []
    for line in graph_lines:
        parts = line.split()
        iata = parts[0]
        nodes.append(iata)
        dot.node(iata, label=iata)

    # Second pass: add edges
    for line in graph_lines:
        parts = line.split()
        src = parts[0]

        # Remaining parts come in triples: dst_nr, km, min
        triples = parts[1:]
        for i in range(0, len(triples), 3):
            dst_nr = int(triples[i])
            km = triples[i + 1]
            minutes = triples[i + 2]

            dst = nodes[dst_nr]
            label = f"{km} km\n{minutes} min"
            dot.edge(src, dst, label=label)

    dot.render(args.output, cleanup=True)
    print(f"Written {args.output}")

    return 0


if __name__ == "__main__":
    raise exit(main())
