#!/usr/bin/env python3
import argparse
import json
import sys
from typing import TextIO


def comma_list(value: str | None) -> set[str] | None:
    if value is None:
        return None

    return {v.strip() for v in value.split(",") if v.strip()}


def parse_args() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="Convert airport dataset JSON into adjacency-list format."
    )
    parser.add_argument(
        "json_path",
        help="Path to ds.json input file",
    )
    parser.add_argument(
        "--continent",
        help="Comma-separated list of continents to include (match 'continent' field)",
    )
    parser.add_argument(
        "--country",
        help="Comma-separated list of countries to include (match 'country_code' field)",
    )
    parser.add_argument(
        "--output",
        help="Output file path (default: stdout)",
    )
    return parser.parse_args()


def airport_matches(
    airport: dict,
    continents: set[str] | None,
    countries: set[str] | None,
) -> bool:
    if continents is not None and airport.get("continent") not in continents:
        return False
    if countries is not None and airport.get("country_code") not in countries:
        return False

    return True


def main() -> int:
    args = parse_args()

    continents = comma_list(args.continent)
    countries = comma_list(args.country)

    with open(args.json_path, "r", encoding="utf-8") as fin:
        data = json.load(fin)

    filtered_airports = {
        iata: airport
        for iata, airport in data.items()
        if airport_matches(airport, continents, countries)
    }

    sorted_airports = sorted(filtered_airports.keys())

    # Map IATA code -> numeric ID
    iata_to_nr = {iata: i for i, iata in enumerate(sorted_airports)}

    n_airports = len(sorted_airports)

    # Build adjacency list
    routes: list[list[tuple[int, int, int]]] = [[] for _ in range(n_airports)]

    for src_iata in sorted_airports:
        src_nr = iata_to_nr[src_iata]
        for route in filtered_airports[src_iata].get("routes", []):
            dst_iata = route["iata"]
            if dst_iata not in iata_to_nr:
                continue

            dst_nr = iata_to_nr[dst_iata]
            km = int(route["km"])
            minutes = int(route["min"])
            routes[src_nr].append((dst_nr, km, minutes))

    n_routes = sum(len(r) for r in routes)

    # Build output
    lines: list[str] = []
    lines.append(f"{n_airports} {n_routes}")

    for src_nr, src_routes in enumerate(routes):
        parts = [sorted_airports[src_nr]]
        for dst_nr, km, minutes in src_routes:
            parts.append(f"{dst_nr} {km} {minutes}")
        lines.append(" ".join(parts))

    output: TextIO
    if args.output:
        output = open(args.output, "w", encoding="utf-8")
    else:
        output = sys.stdout

    try:
        output.write("\n".join(lines))
        output.write("\n")
    finally:
        if output is not sys.stdout:
            output.close()

    return 0


if __name__ == "__main__":
    sys.exit(main())
