#include <Arduino.h>
#include <DFRobotDFPlayerMini.h>
#include <LiquidCrystal_I2C.h>
#include <SPI.h>
#include <MFRC522.h>

#define SS_PIN 10
#define RST_PIN 9
MFRC522 myRFID(SS_PIN, RST_PIN);   // Create MFRC522 instance.

LiquidCrystal_I2C lcd(0x27, 16, 2);  // Create LCD instance.

const int button_start_stop = 2;
boolean currently_playing = false;

String cards_UIDs[10] = {
  "07 03 3B 00",   // card 1
  "D3 33 10 2A",   // card 2
  "03 BA 49 29",   // card 3
  "A3 C8 8D 28",   // card 4
  "0C 6D 49 00",   // card 5
  "BC 12 2F 00",  // card 6
  "61 23 50 00",  // card 7
  "1B EE 36 00",  // card 8
  "F5 95 3A 00",  // card 9
  "FD CE 49 00"   // card 10
};

String previous_card_UID = ""; 
const int volume_pot = A0;
int current_volume_level = 0;
int previous_volume_level = 0;

#if (defined(ARDUINO_AVR_UNO) || defined(ESP8266))   // Using a soft serial port
#include <SoftwareSerial.h>
SoftwareSerial softSerial(/*rx =*/6, /*tx =*/5);
#define FPSerial softSerial
#else
#define FPSerial Serial1
#endif

DFRobotDFPlayerMini myDFPlayer;  // Create a DFPlayerMini instance.

void setup() {
  // put your setup code here, to run once:
  Serial.begin(9600);
  Serial.println("JukeBox");

  // DFPlayer
  #if (defined ESP32)
  FPSerial.begin(9600, SERIAL_8N1, /*rx =*/D3, /*tx =*/D2);
  #else
  FPSerial.begin(9600);
  #endif

  myDFPlayer.begin(FPSerial);
  current_volume_level = map(analogRead(volume_pot), 0, 1023, 0, 30);   //scale the pot value and volume level
  myDFPlayer.volume(current_volume_level);
  previous_volume_level = current_volume_level;

  // LCD
  lcd.init();       // Initiate LCD.
  lcd.backlight();     // Turn backlight on.
  lcd.clear();        // Clar the LCD.
  lcd.setCursor(0, 0);  // Set the cursor at the beginning.
  lcd.print("Choose a card.");

  // MFRC5222
  SPI.begin();      // Initiate  SPI bus
  myRFID.PCD_Init();   // Initiate MFRC522

  // button_start-stop
  pinMode(button_start_stop, INPUT_PULLUP);

}

void loop() {
  //scale the pot value and volume level
  current_volume_level = map(analogRead(volume_pot), 0, 1023, 0, 30);
  // if the volume it's different from last time, change it
  if (current_volume_level != previous_volume_level) {
    myDFPlayer.volume(current_volume_level);
  }
  previous_volume_level = current_volume_level;

  bool last_button_state = HIGH;
  bool button_state = digitalRead(button_start_stop);
  delay(150); // debounce button

  // See if the button is pressed or not
  if (button_state == LOW && last_button_state == HIGH) {
    delay(50);  // Debounce
    if (currently_playing) { // when playing, stop it
      myDFPlayer.pause();
      currently_playing = false;
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Stopped song.");
    } else {
      myDFPlayer.start(); // else, start(resume) the song
      currently_playing = true;
      lcd.clear();
      lcd.setCursor(0, 0);
      lcd.print("Playing...");
    }
  }
  last_button_state = button_state;


  // wait for RFID cards to be scanned
  if ( ! myRFID.PICC_IsNewCardPresent() || !myRFID.PICC_ReadCardSerial()) {
    return;
  } else {
    // card scanned - take the UID
    String current_card_UID = "";
    for (byte i = 0; i < myRFID.uid.size; i++) {
      if (myRFID.uid.uidByte[i] < 0x10) {
        current_card_UID += "0";
      }
      current_card_UID += String(myRFID.uid.uidByte[i], HEX);
      if (i < myRFID.uid.size - 1) {
        current_card_UID += " ";
      }
    }
    current_card_UID.toUpperCase();

    // play the song associated to the card
    for (int i = 0; i < 10; i++) {
      if (current_card_UID == cards_UIDs[i]) {
        // a new card has been scanned, so we stop the previous song and play the new one
        if (current_card_UID != previous_card_UID) {
          previous_card_UID = current_card_UID;
          myDFPlayer.stop(); // stop the current song
          lcd.clear();
          lcd.setCursor(0, 0);
          lcd.print("Interrupted song.");
          currently_playing = false;
          delay(300);
          myDFPlayer.play(i + 1);
          currently_playing = true;
          lcd.clear();
          lcd.setCursor(0, 0);
          lcd.print("Playing song ");
          lcd.setCursor(0, 1);
          lcd.print("from card ");
          lcd.print(i + 1);
        } else {
          Serial.println("Same song.");
        }
      }
    }

    myRFID.PICC_HaltA();  // Stop reading
  }
}
