#include "Arduino.h"
#include <SoftwareSerial.h>
#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <Servo.h>

// Limitele de miscare ale degetelor, respectiv, incheieturii.
#define FINGERS_LOW_LIMIT 0
#define FINGERS_HIGH_LIMIT 90

#define WRIST_LOW_LIMIT 0
#define WRIST_HIGH_LIMIT 180

// Pinii prin care vom comunica cu modulul de bluetooth
// care la randul lui comunica cu aplicatia instalata
// in telefon
const byte rxPin = 9;
const byte txPin = 8;

// Pinii prin intermediul carora
// vom controla cele 2 servomotoare.
const byte wristServoPin = 10;
const byte fingersServoPin = 11;

// Constante ce tin de comunicarea prin bluetooth si "protocolul" pe care
// il folosim.
// Marcheaza inceputul unui mesaj primit prin bluetooth.
const char START_TOKEN = '?';
// Marcheaza sfarsitul unui mesaj.
const char END_TOKEN = ';';
// Pentru a putea distinge 2 valori diferite pe care
// le primitem(una pentru unghiul degetelor si una
// pentru unghiul incheieturii).
const char DELIMIT_TOKEN = '&';
// Cand primim un numar mai mare de 20 de caractere
// prin bluetooth, presupunem ca a fost o problema la comunicarea
// si vom ignora mesajul primit.
const int CHAR_TIMEOUT = 20;
// 2 String-uri ce tin de formatul in care am primit datele.
String fingersFormat = "fingers=";
String wristFormat = "wrist=";
// Flag care marcheaza faptul ca se asteapta inceperea unui nou mesaj.
bool waitingForStartToken = true;
String messageBuffer = "";
// Valorile extrase din mesaj salvate ca String-uri.
String fingersValueString = "";
String wristValueString = "";

// Valorile scrise pe servomotoare la momentul curent.
int wristAngle = 0;
int fingersAngle = 0;

// Valorile primite prin bluetooth de la telefon, practic sunt string-urile
// de mai sus convertite la int.
int wristValueInt = 0;
int fingersValueInt = 0;

// Seriala prin care vom comunica cu modului de Bluetooth HC-05, care la randul lui
// este conectat la aplicatia de pe telefon.
SoftwareSerial mySerial(rxPin, txPin); // RX TX

// LCD-ul pe care vom afisa mesaje.
LiquidCrystal_I2C lcd(0x27, 16, 2);

// Servomotorul care va simula miscarea incheieturii.
Servo wristServo;
// Servomotorul care va simula miscarea degetelor.
Servo fingersServo;

// Semnalul trimis catre telefon.
bool outputValue = false;
// Flag prin care ne dam seama daca a trecut o secunda
// si putem realiza operatiile specificate.
bool flag = false;
// Counter folosit in cadrul intreruperii pentru a vedea
// cand s-a intrat de 250 de ori in intrerupere, deci
// a trecut o secunda.
long one_sec_counter = 0;

// Functia de tratare a intreruperilor generate de Timer0.
ISR(TIMER0_COMPA_vect) {
  if(one_sec_counter  == 250) {
    // Marcam ca a trecut o secunda deci se pot face operatiile
    // de actualizare a LCD-ului si transmitere semnal catre telefon.
    flag = true;
    one_sec_counter = 0;
  } else {
    one_sec_counter ++;

    flag = false;
  }
}

/*
 * Functie de initializare a Timer0.
 */
void init_timer0_one_sec() {
  // Configuram Timer0 in modul CTC.
  TCCR0A = (1 << WGM01);

  // Setam un prescaler de 256
  TCCR0B = (1 << CS02);

  // Un prag de 250.
  OCR0A = 249;

  // Practic obtinem o frecventa a intreruperii de: 16MHz / 256(prescaler) = 62500 / 250(OCR0A + 1) =
  // = 250Hz, deci, practic, ca sa obtinem o frecventa de 1 secunda cum vrem noi, ne mai trebuie
  // un counter auxiliar care sa mearga pana la 250, si atunci sa faca ce ne dorim noi(mai exact
  // sa trimita un semnal catre placuta) si dupa sa il resetam si sa asteptam iar sa ajunga la valoarea
  // stabilita.

  // Activam intreruperea atunci cand se ajunge cu counter-ul
  // la OCR0A.
  TIMSK0 = (1 << OCIE0A);
}

/*
 * Trimitem un semnal(0/1) catre telefon pentru ca utilizatorul
 * sa vada in cadrul aplicatiei ca are inca o conexiune stabila
 * cu placuta.
 */
void send_signal_to_phone() {
  outputValue = !outputValue;
  if(outputValue) {
    mySerial.write("1");
  } else {
    mySerial.write("0");
  }
}

/*
 * Functie care actualizeaza informatiile afisate pe LCD 
 */
void print_info_on_lcd() {
  lcd.clear();
    
  lcd.setCursor(3, 0);
  lcd.print("Bionic Hand");

  if(fingersAngle == FINGERS_HIGH_LIMIT) {
    lcd.setCursor(0, 1);
    lcd.print("FINGERS AT LIMIT!");
  } else if(wristAngle == WRIST_HIGH_LIMIT) {
    lcd.setCursor(0, 1);
    lcd.print("WRIST AT LIMIT!"); 
  } else {
    lcd.setCursor(4, 1);
    lcd.print("f=" + String(fingersAngle) + ";w=" + String(wristAngle));
  }
}

/*
 * Functie care verifica daca se primeste un mesaj prin seriala de comunicare
 * cu modulul de bluetooth, in cazul in care se primeste un mesaj complet acesta
 * este parsat prin apelul functiei special definite.
 */
void verify_bt_serial() {
  char nextData;

  if(mySerial.available()) {
    // Verificam daca urmeaza sa inceapa un mesaj
    if(waitingForStartToken) {
      do {
        nextData = mySerial.read();
      } while(nextData != START_TOKEN && mySerial.available());
      if (nextData == START_TOKEN) {
        Serial.println("message start");
        // Asteptam sa primim intregul mesaj.
        waitingForStartToken = false;
      }
    }

    // Citim comanda
    if(!waitingForStartToken && mySerial.available()) {
      do {
        nextData = mySerial.read();
        messageBuffer += nextData;
      } while(nextData != END_TOKEN && mySerial.available());
    }

    // Verificam daca mesajul s-a primit in totalitate.
    if(nextData == END_TOKEN) {
      // Stergem ultimul caracter(';').
      messageBuffer = messageBuffer.substring(0, messageBuffer.length() - 1);
      Serial.println("message complete - " + messageBuffer);
      // Parsam mesajul primit de la telefon.
      parse_message();
      // Resetam continutul buffer-ului.
      messageBuffer = "";
      // Marcam faptul ca asteptam alt mesaj.
      waitingForStartToken = true;
    }

    // Verificam daca un mesaj contine mai multe caractere decat permitem noi
    // inseamna ca ceva a mers prost cu receptia/transmisia si vom ignora
    // mesajul.
    if(messageBuffer.length() > CHAR_TIMEOUT) {
      Serial.println("message data timeout - " + messageBuffer);
      messageBuffer = "";
      waitingForStartToken = true;
    }
  }
}

/*
 * Functie care parseaza un mesaj de tipul fingers=xx&wrist=yyy primit 
 * de la telefon prin bluetooth.
 */
void parse_message() {
  // Verifica daca mesajul incepe cum ne-am dori.
  if(messageBuffer.startsWith(fingersFormat)){
    // Inseamna ca avem un mesaj de tipul fingers=xx&wrist=yyy
    // In continuare va trebuie sa extragem din mesaj valorile
    // celor doua unghiuri
    
    if(messageBuffer.indexOf(wristFormat) == -1) {
      return;
    }
    
    // Acum stim sigur ca daca s-a ajung in acest punct mesajul contine si valoarea pentru wrist angle.
    // Extragem string-ul care contine unghiul pentru fingers.
    int indexOfDelimiter = messageBuffer.indexOf(DELIMIT_TOKEN);
    fingersValueString = messageBuffer.substring(fingersFormat.length(), indexOfDelimiter);

    // Extragem string-ul care contine unghiul pentru wrist.
    wristValueString = messageBuffer.substring(indexOfDelimiter + 1 + wristFormat.length());

    // Scriem valorile primite(daca sunt noi) pe cele doua servo-motoare.
    write_values_to_servo();
  }
}

/*
 * Functie care valideaza valorile primite prin bluetooth si in cazul in care
 * acestea sunt in limite sau veridice, atunci da comanda catre cele 2 motorase.
 */
void write_values_to_servo() {
  fingersValueInt = (int) fingersValueString.toInt();
  if(fingersValueInt == 0 && !(fingersAngle >=0 && fingersAngle <= 5)) {
    // Daca valoarea pe care am primit-o prin Bluetooth este 0, si vechea valoarea pe care
    // am salvat-o in unghiul pentru fingers nu este 0 sau 1(adica nu ne-am intors cu
    // slider-ul la 0 din 1) inseamna ca a mers ceva prost la parsare, adica
    // este posibil ca dupa fingers= sa fi primit niste garbage si trebuie
    // sa tinem cont de acest lucru pentru ca ar fi un glitch, in cazul in care
    // noi sa presupunem ca am fi cu unghiul la 75 de grade, primim o valoare garbage
    // prin conexiune, deci fingerValueInt va fi 0, si ne-am intoarce cu degetele
    // de al 75 de grade la 0, ceea ce nu ne-am dori. Deci in acest caz nu vom face
    // write pe servomotorul pentru fingers, adica nu vom actualiza valoarea fingersAngle.
    // In plus se mai verifica si daca noua valoarea este egala cu cea veche pentru a nu
    // face write degeaba la aceeasi valoare.
  } else {
    if(fingersValueInt >= FINGERS_LOW_LIMIT 
       && fingersValueInt <= FINGERS_HIGH_LIMIT
       && fingersValueInt != fingersAngle) {
      fingersAngle = fingersValueInt;
      Serial.println(fingersAngle);
      fingersServo.write(fingersAngle);
    }
  }
  
  wristValueInt = (int) wristValueString.toInt();
  // Facem aceeasi verificare ca si la fingers.
  if(wristValueInt == 0 && !(wristAngle >=0 && wristAngle <= 5)) {
  } else {
    if(wristValueInt >= WRIST_LOW_LIMIT 
      && wristValueInt <= WRIST_HIGH_LIMIT
      && wristValueInt != wristAngle) {
      wristAngle = wristValueInt;
      Serial.println(wristAngle);
      wristServo.write(wristAngle);
    }
  } 
}

void setup() {
  // Initializam timer-ul de o secunda.
  init_timer0_one_sec();

  // Dam enable la intreruperile globale.
  sei();
  
  // Configuram baudrate-ul serialei cu modulul de Bluetooth HC-05
  mySerial.begin(9600);
  Serial.begin(9600);

  // Configuram LCD-ul.
  lcd.init();
  lcd.backlight();
  
  // Configuram servomotorul incheieturii.
  wristServo.attach(wristServoPin);
  wristServo.write(wristAngle);

  // Configuram servomotorul degetelor.
  fingersServo.attach(fingersServoPin);
  fingersServo.write(fingersAngle);
}

void loop() {
  // Verificam mereu daca am primit informatii prin seriala asociata
  // cu modulul de Bluetooth HC-05.
  verify_bt_serial();

  // Flag-ul care este true odata la o secunda si prin intermediul
  // caruia stim cand trebuie sa actualizam informatiile de pe LCD
  // si sa trimitem semnal catre telefon. 
  if(flag == true) {
    // Apelam functia de actualizare a mesajelor de pe LCD.
    print_info_on_lcd();
    // Apelam functia de transmis un semnal de 1 byte catre telefon.
    send_signal_to_phone();  
  }
}
