/* ------- libraries ------- */
#include <Wire.h>

#include <LiquidCrystal_I2C.h>

/* ------- macros --------*/

#define PUMP_PIN 6
#define SOIL_PIN A0
#define BUTTON_PIN 2

#define LED_RED_PIN 3
#define LED_GREEN_PIN 4
#define LED_BLUE_PIN 5

#define DRY_SOIL 0
#define DAMP_SOIL 1

/* ------- global variables ------*/
LiquidCrystal_I2C lcd(0x27, 16, 2);
int soil_hum_percentage = -1;
int good_humidity = -1;
int pressed_button = -1;


// under this treshold the plant will be watered
#define SOIL_WATERING_TRESHOLD_PERCENTAGE 50



/* ----------------------------------------- setup  -----------------------------------------*/

void setup_lcd() {
  lcd.init();
  lcd.backlight();
  //lcd.clear();

}

void setup_led() {
  pinMode(LED_RED_PIN, OUTPUT);
  pinMode(LED_GREEN_PIN, OUTPUT);
  pinMode(LED_BLUE_PIN, OUTPUT);
}

void setup_pump() {
  pinMode(PUMP_PIN, OUTPUT);
}

void setup_button(){
  pinMode(BUTTON_PIN, INPUT_PULLUP);
}


void setup() {
  // put your setup code here, to run once:

  setup_lcd();
  setup_led();
  setup_pump();
  setup_button();

  Serial.begin(9600);

}



/* ---------------------------- helper functions -----------------------------------------*/

// Reads and updates current soil level
void read_soil_humidity() {
  // Measurements
  // air: ~ 1023
  // water: ~ 100 
  // dry soil: 1022 
  // damp soil: 120 - 250
  
  
  int sensor_value = analogRead(SOIL_PIN);
  
  // sensor interval:  [0, 1023]
  soil_hum_percentage = 100 - map(sensor_value, 0, 1023, 0, 100);


  if(soil_hum_percentage < SOIL_WATERING_TRESHOLD_PERCENTAGE){
    good_humidity = 0;
  }else{
    good_humidity = 1;
  }

}



//Turns the led green if the soil has a proper humidity
//Turns the led blue if the plant is watered (automatic or manual)
void check_soil_hum_led(){

  if(good_humidity == 0 || pressed_button == 1){
    //BLUE
    digitalWrite(LED_RED_PIN, LOW);
    digitalWrite(LED_GREEN_PIN, LOW);
    digitalWrite(LED_BLUE_PIN, HIGH);
  }else if (good_humidity == 1 && pressed_button == 0) {
    //GREEN
    digitalWrite(LED_RED_PIN, LOW);
    digitalWrite(LED_GREEN_PIN, HIGH);
    digitalWrite(LED_BLUE_PIN, LOW);
  }else{
    digitalWrite(LED_RED_PIN, HIGH);
    digitalWrite(LED_GREEN_PIN, LOW);
    digitalWrite(LED_BLUE_PIN, LOW);
  }
}


//If the button is pressed, the water pump will be turned on
//so the plant will be watered manually 
void button_check(){
  int digitalValue = digitalRead(BUTTON_PIN);

  if(digitalValue == HIGH){
    pressed_button = 1;
  }else{
    pressed_button = 0;
  }
}


// Turns on the pump if soil humidity is too low
void operate_pump() {
  // Check if soil humidity has fallen under 50%
  
  if (good_humidity == 0 || pressed_button == 1) { //the plant is watered
    digitalWrite(PUMP_PIN, HIGH);
  } else {
    digitalWrite(PUMP_PIN, LOW);
  }
}

// Prints the soil humidity percentage on the lcd
void print_lcd(int percentage) {
  lcd.setCursor(1,0);
  lcd.print("Soil humidity");
  
  lcd.setCursor(6, 1);
  lcd.print(percentage);
  lcd.print("%");
}


void loop() {
  // put your main code here, to run repeatedly:

  button_check();

  read_soil_humidity();

  print_lcd(soil_hum_percentage);

  check_soil_hum_led();
  
  operate_pump();

}
