package platform;

import song.Song;
import user.User;
import utils.JSONReader;

import javax.sound.sampled.*;
import javax.swing.*;
import javax.swing.border.EmptyBorder;
import java.awt.*;
import java.io.File;
import java.io.IOException;
import java.util.List;

public class Spotify implements Subject {
    private List<User> users;
    private List<Song> songs;
    private List<Observer> observers;

    // TODO 1: Implement Singleton pattern
    // - Make constructor private
    // - Add: private static Spotify instance;
    // - Add: public static Spotify getInstance() { ... }
    // - Instantiate the lists (users, songs, observers) in the private constructor

    public void addUser(User user) {
        users.add(user);
    }

    public void addSong(Song song) {
        songs.add(song);
        // TODO 5: Notify observers when a new song is added
        // notifyObservers("New song added: " + song.getTitle());
    }

    // TODO 5: Implement Subject interface methods
    @Override
    public void addObserver(Observer o) {
        // Add observer to the list
    }

    @Override
    public void removeObserver(Observer o) {
        // Remove observer from the list
    }

    @Override
    public void notifyObservers(String message) {
        // Iterate through observers and call update(message)
    }

    public List<User> getUsers() {
        return users;
    }

    public List<Song> getSongs() {
        return songs;
    }

    public void readData() throws IOException {
        List<User> loadedUsers = JSONReader.readUsers(
                "src/main/java/users.json"
        );

        users.addAll(loadedUsers);

        // TODO 5: Add loaded users as observers
        // for (User user : loadedUsers) { addObserver(user); }

        songs.addAll(JSONReader.readSongs(
                "src/main/java/songs.json",
                loadedUsers
        ));
        System.out.println("Data loaded successfully.");
    }

    public Double calculateRevenue() {
        // TODO 6: Use Visitor pattern to compute total revenue
        // RevenueVisitor visitor = new RevenueVisitor();
        // for (User user : users) { user.accept(visitor); }
        // return visitor.getTotalRevenue();
        return 0.0;
    }

    // ==================== GUI - DON'T MODIFY ====================
    public void run() {
        SwingUtilities.invokeLater(() -> {
            JFrame frame = new JFrame("Spotify - Music Player");
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.setSize(650, 450);
            frame.setLocationRelativeTo(null);

            // Song list
            DefaultListModel<String> listModel = new DefaultListModel<>();
            for (Song song : songs) {
                listModel.addElement(song.getTitle());
            }
            JList<String> songList = new JList<>(listModel);
            songList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
            songList.setFont(new Font("Arial", Font.PLAIN, 14));
            JScrollPane scrollPane = new JScrollPane(songList);

            // Status label
            JLabel statusLabel = new JLabel("Select a song and press Play");
            statusLabel.setBorder(new EmptyBorder(10, 10, 10, 10));
            statusLabel.setFont(new Font("Arial", Font.BOLD, 14));

            // Buttons
            JButton playButton = new JButton("Play");
            JButton pauseButton = new JButton("Pause");
            JButton stopButton = new JButton("Stop");

            JPanel buttonPanel = new JPanel(new FlowLayout());
            buttonPanel.add(playButton);
            buttonPanel.add(pauseButton);
            buttonPanel.add(stopButton);

            // Progress bar
            JProgressBar progressBar = new JProgressBar(0, 100);
            progressBar.setValue(0);
            progressBar.setStringPainted(true);
            progressBar.setString("00:00 / 00:00");

            // Time label
            JLabel timeLabel = new JLabel("Duration: --:--");
            timeLabel.setHorizontalAlignment(SwingConstants.CENTER);

            // Bottom panel
            JPanel bottomPanel = new JPanel(new BorderLayout());
            bottomPanel.add(buttonPanel, BorderLayout.NORTH);
            bottomPanel.add(progressBar, BorderLayout.CENTER);
            bottomPanel.add(timeLabel, BorderLayout.SOUTH);
            bottomPanel.setBorder(new EmptyBorder(5, 10, 10, 10));

            // Layout
            frame.setLayout(new BorderLayout());
            frame.add(statusLabel, BorderLayout.NORTH);
            frame.add(scrollPane, BorderLayout.CENTER);
            frame.add(bottomPanel, BorderLayout.SOUTH);

            // Audio player state
            final Clip[] currentClip = {null};
            final Timer[] progressTimer = {null};
            final boolean[] isPaused = {false};

            // Play button action
            playButton.addActionListener(e -> {
                String selected = songList.getSelectedValue();
                if (selected == null) {
                    statusLabel.setText("Please select a song first!");
                    return;
                }

                // If paused, resume
                if (isPaused[0] && currentClip[0] != null) {
                    currentClip[0].start();
                    isPaused[0] = false;
                    statusLabel.setText("Playing: " + selected);
                    if (progressTimer[0] != null) progressTimer[0].start();
                    return;
                }

                // Stop current playback
                if (currentClip[0] != null) {
                    currentClip[0].stop();
                    currentClip[0].close();
                    if (progressTimer[0] != null) progressTimer[0].stop();
                }

                // Find audio file (WAV only)
                File songsFolder = new File("songs");
                File[] matches = songsFolder.listFiles((dir, name) ->
                    name.toLowerCase().contains(selected.toLowerCase()) &&
                    name.endsWith(".wav")
                );

                if (matches == null || matches.length == 0) {
                    statusLabel.setText("Audio file not found for: " + selected);
                    progressBar.setValue(0);
                    progressBar.setString("No file");
                    return;
                }

                File audioFile = matches[0];

                try {
                    AudioInputStream audioStream = AudioSystem.getAudioInputStream(audioFile);
                    currentClip[0] = AudioSystem.getClip();
                    currentClip[0].open(audioStream);

                    long totalMicroseconds = currentClip[0].getMicrosecondLength();
                    int totalSeconds = (int) (totalMicroseconds / 1_000_000);
                    String totalTime = String.format("%02d:%02d", totalSeconds / 60, totalSeconds % 60);
                    timeLabel.setText("Duration: " + totalTime);

                    // Progress timer
                    progressTimer[0] = new Timer(500, evt -> {
                        if (currentClip[0] != null && currentClip[0].isRunning()) {
                            long currentMicro = currentClip[0].getMicrosecondPosition();
                            int progress = (int) ((currentMicro * 100) / totalMicroseconds);
                            progressBar.setValue(progress);

                            int currentSec = (int) (currentMicro / 1_000_000);
                            String currentTime = String.format("%02d:%02d", currentSec / 60, currentSec % 60);
                            progressBar.setString(currentTime + " / " + totalTime);
                        }
                    });

                    // When song ends
                    currentClip[0].addLineListener(event -> {
                        if (event.getType() == LineEvent.Type.STOP && !isPaused[0]) {
                            SwingUtilities.invokeLater(() -> {
                                if (currentClip[0] != null &&
                                    currentClip[0].getMicrosecondPosition() >= currentClip[0].getMicrosecondLength() - 100000) {
                                    statusLabel.setText("Finished: " + selected);
                                    progressBar.setValue(100);
                                    progressBar.setString(totalTime + " / " + totalTime);
                                    if (progressTimer[0] != null) progressTimer[0].stop();
                                }
                            });
                        }
                    });

                    currentClip[0].start();
                    progressTimer[0].start();
                    isPaused[0] = false;
                    statusLabel.setText("Playing: " + selected);

                } catch (UnsupportedAudioFileException ex) {
                    statusLabel.setText("Unsupported format (need WAV): " + audioFile.getName());
                } catch (LineUnavailableException ex) {
                    statusLabel.setText("Audio device unavailable");
                } catch (IOException ex) {
                    statusLabel.setText("Error reading file: " + ex.getMessage());
                }
            });

            // Pause button action
            pauseButton.addActionListener(e -> {
                if (currentClip[0] != null && currentClip[0].isRunning()) {
                    currentClip[0].stop();
                    isPaused[0] = true;
                    if (progressTimer[0] != null) progressTimer[0].stop();
                    statusLabel.setText("Paused");
                }
            });

            // Stop button action
            stopButton.addActionListener(e -> {
                if (currentClip[0] != null) {
                    currentClip[0].stop();
                    currentClip[0].setMicrosecondPosition(0);
                    isPaused[0] = false;
                    if (progressTimer[0] != null) progressTimer[0].stop();
                    progressBar.setValue(0);
                    progressBar.setString("00:00 / 00:00");
                    statusLabel.setText("Stopped");
                }
            });

            // Clean up on close
            frame.addWindowListener(new java.awt.event.WindowAdapter() {
                @Override
                public void windowClosing(java.awt.event.WindowEvent e) {
                    if (currentClip[0] != null) {
                        currentClip[0].stop();
                        currentClip[0].close();
                    }
                    if (progressTimer[0] != null) {
                        progressTimer[0].stop();
                    }
                }
            });

            frame.setVisible(true);
        });
    }
}
