#include <Wire.h>
#include <LiquidCrystal_PCF8574.h>
#include <SPI.h>
#include <MFRC522.h>
#include <Servo.h>
#include <avr/io.h>  

// PINOUT
#define SERVO_PIN      3
#define TRIG1_PIN      4   // intrare ultrasonic
#define ECHO1_PIN      9
#define TRIG2_PIN      6   // iesire ultrasonic
#define ECHO2_PIN      7
#define RFID_SS_PIN   10   // CS
#define RFID_RST_PIN   8   // RST

// PARAMETRII
const long    thresholdCm      = 10;        // distanta sub care consideram obstacol
const uint8_t barrierDown      = 180;       // pozitia servomotorului jos
const uint8_t barrierUp        = 360;       // pozitia servomotorului sus
const int     maxSpots         = 3;         // nr. total locuri
const unsigned long carClearTime   = 1500;  // ms pana consideram ca masina a plecat de la senzor
const unsigned long minRaiseInterval = 10000; // 10 s intre doua ridicari succesive (intrare sau iesire)

// OBIECTE 
Servo               barrierServo;
MFRC522             mfrc522(RFID_SS_PIN, RFID_RST_PIN);
LiquidCrystal_PCF8574 lcd(0x27);

// UID-uri autorizate ale cartelelor
const byte authorizedUID[][4] = {
  { 0x90, 0x17, 0x79, 0x14 },
  { 0xA8, 0xCC, 0x2C, 0x12 }
};
const size_t authorizedCount = sizeof(authorizedUID)/sizeof(authorizedUID[0]);

// STARI
bool authorized    = false;
bool barrierIsUp   = false;
bool exiting       = false;
int  carsInside    = 0;
unsigned long lastRaiseMillis = 0;   // momentul ultimei ridicari a barierei (intrare sau iesire)

// UTILITY
long readDistanceCm(int trigPin, int echoPin) {
  digitalWrite(trigPin, LOW);
  delayMicroseconds(2);
  digitalWrite(trigPin, HIGH);
  delayMicroseconds(10);
  digitalWrite(trigPin, LOW);
  long dur = pulseIn(echoPin, HIGH, 30000);
  return (dur>0) ? dur*0.034/2 : 999;
}

bool isAuthorized(const byte *uid, byte uidSize) {
  for (size_t i=0; i<authorizedCount; i++) {
    bool match = true;
    for (byte j=0; j<uidSize; j++){
      if (uid[j] != authorizedUID[i][j]) { match = false; break; }
    }
    if (match) return true;
  }
  return false;
}

// LED-uri prin PORTC, cu set/clear explicit
void updateLEDs() {
  PORTC &= ~((1 << PC0) | (1 << PC1) | (1 << PC2)); // stinge toate

  if (carsInside == 0)            PORTC |= (1 << PC0);   // verde
  else if (carsInside < maxSpots) PORTC |= (1 << PC1);   // galben
  else                            PORTC |= (1 << PC2);   // rosu
}

void showMessage(const char *msg) {
  lcd.clear();
  lcd.setCursor(0, 0);
  lcd.print(msg);
}

void setup() {
  Serial.begin(9600);

  // LED-uri
  DDRC |= (1 << PC0) | (1 << PC1) | (1 << PC2);
  PORTC &= ~((1 << PC0) | (1 << PC1) | (1 << PC2));
  updateLEDs();

  // LCD I2C
  Wire.begin();
  lcd.begin(16, 2);
  lcd.setBacklight(true);
  showMessage("Welcome!");
  delay(1500);
  showMessage("Scan card...");

  // Servo
  barrierServo.attach(SERVO_PIN);
  barrierServo.write(barrierDown);

  // Ultrasonice
  pinMode(TRIG1_PIN, OUTPUT);
  pinMode(ECHO1_PIN, INPUT);
  pinMode(TRIG2_PIN, OUTPUT);
  pinMode(ECHO2_PIN, INPUT);

  // RFID + SPI
  SPI.begin();
  mfrc522.PCD_Init();
}

void loop() {
  // INTRARE CU CARD 
  if (mfrc522.PICC_IsNewCardPresent() && mfrc522.PICC_ReadCardSerial()) {
    if (carsInside >= maxSpots) {
      showMessage("Parcare plina");
      updateLEDs();
      delay(2000);
      showMessage("Scan card...");
    }
    else if (isAuthorized(mfrc522.uid.uidByte, mfrc522.uid.size)) {
      authorized = true;
      showMessage("Cartela corecta");
      delay(1000);
    }
    else {
      authorized = false;
      showMessage("Cartela incorecta");
      delay(2000);
      showMessage("Scan card...");
    }
    mfrc522.PICC_HaltA();
    delay(200);
  }

  // PROCESUL DE INTRARE
  if (authorized) {
    long d1 = readDistanceCm(TRIG1_PIN, ECHO1_PIN);
    long d2 = readDistanceCm(TRIG2_PIN, ECHO2_PIN);
    // ridicam bariera numai daca:
    //  - este jos
    //  - masina este prezenta la intrare
    //  - au trecut minRaiseInterval de la ultima ridicare
    if (!barrierIsUp && d1 <= thresholdCm && (millis() - lastRaiseMillis >= minRaiseInterval)) {
      barrierServo.write(barrierUp);
      barrierIsUp = true;
      lastRaiseMillis = millis();
      delay(800);
    }
    // dupa ce masina a depasit senzorul de iesire, coboram bariera si resetam autorizarea
    else if (barrierIsUp && d2 <= thresholdCm) {
      barrierServo.write(barrierDown);
      barrierIsUp = false;
      authorized = false;
      carsInside++;
      updateLEDs();
      showMessage("Scan card...");
      delay(800);
    }
  }

  // IESIRE AUTOMATA 
  // ridicam bariera la iesire doar daca:
  //  - este jos
  //  - există masini in parcare
  //  - senzorul de iesire detecteaza vehicul
  //  - au trecut minRaiseInterval de la ultima ridicare
  if (!authorized && !barrierIsUp && carsInside > 0) {
    long d2 = readDistanceCm(TRIG2_PIN, ECHO2_PIN);
    if (d2 <= thresholdCm && (millis() - lastRaiseMillis >= minRaiseInterval)) {
      barrierServo.write(barrierUp);
      barrierIsUp = true;
      exiting = true;
      lastRaiseMillis = millis();   // retinem momentul ridicarii
      showMessage("Iesire...");
      delay(800);
    }
  }

  // după ce masina a trecut inchidem bariera si actualizam contorul
  if (exiting && barrierIsUp) {
    long d2 = readDistanceCm(TRIG2_PIN, ECHO2_PIN);
    if (d2 > thresholdCm) {
      unsigned long t0 = millis();
      while (millis() - t0 < carClearTime) {
        if (readDistanceCm(TRIG2_PIN, ECHO2_PIN) <= thresholdCm) t0 = millis();
      }
      barrierServo.write(barrierDown);
      barrierIsUp = false;
      exiting = false;
      if (carsInside > 0) carsInside--;
      updateLEDs();
      showMessage("Drum bun!");
      delay(800);
      showMessage("Scan card...");
    }
  }
}

