from flask import Flask, request, render_template_string, send_from_directory
from datetime import datetime
import os
import platform
import subprocess

app = Flask(__name__)

latest_data = {
    "temp": "N/A",
    "hum": "N/A",
    "rain_detected": False,
    "risk": 0,
    "time": "-",
    "alert_sent": False
}


def send_notification(title, message):
    try:
        system = platform.system()
        if system == "Linux":
            subprocess.call(['notify-send', title, message])
        elif system == "Darwin":  # macOS
            subprocess.call(['osascript', '-e', f'display notification "{message}" with title "{title}"'])
        elif system == "Windows":
            from win10toast import ToastNotifier
            toaster = ToastNotifier()
            toaster.show_toast(title, message, duration=10)
        print(f"Notificare trimisă: {title} - {message}")
    except Exception as e:
        print(f"Eroare la trimiterea notificării: {e}")


HTML_TEMPLATE = """
<!doctype html>
<html lang="ro">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>RoAlert - Monitor Meteo</title>
  <style>
    body {
      font-family: Arial, sans-serif;
      background-color: {% if risk >= 50 %}#ffcccc{% else %}#ccffcc{% endif %};
      color: #333;
      padding: 30px;
      text-align: center;
    }
    .emoji {
      font-size: 50px;
    }
    .alert {
      color: red;
      font-weight: bold;
    }
    .play-button {
      margin-top: 20px;
      padding: 10px 20px;
      background-color: #ff4444;
      color: white;
      border: none;
      border-radius: 5px;
      font-size: 16px;
      cursor: pointer;
    }
  </style>
</head>
<body>
  <h1>📡 RoAlert - Stație Meteo</h1>
  <p><span class="emoji">🌡️</span> Temperatură: <strong>{{ temp }} °C</strong></p>
  <p><span class="emoji">💧</span> Umiditate: <strong>{{ hum }} %</strong></p>
  <p><span class="emoji">🌧️</span> Ploaie: <strong>{{ "DA" if rain_detected else "NU" }}</strong></p>
  <p><span class="emoji">⚠️</span> Risc furtună: <strong>{{ risk }}%</strong></p>
  <p>🕒 Ultima actualizare: {{ time }}</p>

  {% if risk >= 50 %}
    <p class="alert">🔊 ALERTĂ METEO! Condiții severe detectate!</p>
    <button id="playAlertButton" class="play-button">
      APASĂ PENTRU ALERTĂ SONORĂ
    </button>
    <audio id="alertAudio">
      <source src="/alert.mp3" type="audio/mpeg">
      Browserul tău nu suportă audio HTML5.
    </audio>
    <script>
      document.addEventListener('DOMContentLoaded', function() {
        const audio = document.getElementById("alertAudio");
        const playButton = document.getElementById("playAlertButton");

        //redarea automată
        audio.play().catch(e => {
          console.log("Autoplay blocat. Folosește butonul pentru a activa sunetul.");
        });

        // Eveniment pentru butonul de redare
        playButton.addEventListener('click', function() {
          audio.play();

          // Voce TTS
          try {
            const synth = window.speechSynthesis;
            const msg = new SpeechSynthesisUtterance("Atenție! Alertă meteo detectată. Pericol de furtună iminent.");
            msg.lang = "ro-RO";
            synth.speak(msg);
          } catch (e) {
            console.error("Eroare TTS:", e);
          }
        });

        try {
          const synth = window.speechSynthesis;
          const msg = new SpeechSynthesisUtterance("Atenție! Alertă meteo detectată. Pericol de furtună iminent.");
          msg.lang = "ro-RO";
          synth.speak(msg);
        } catch (e) {
          console.error("Eroare TTS automată:", e);
        }
      });
    </script>
  {% endif %}
</body>
</html>
"""


@app.route('/')
def index():
    return render_template_string(HTML_TEMPLATE, **latest_data)


@app.route('/data', methods=['POST'])
def receive_data():
    global latest_data
    data = request.get_json()

    # datele noi
    new_data = {
        "temp": data.get("temp", "N/A"),
        "hum": data.get("hum", "N/A"),
        "rain_detected": bool(data.get("rain_detected", 0)),
        "risk": data.get("risk", 0),
        "time": datetime.now().strftime("%H:%M:%S")
    }


    risk_value = new_data["risk"]
    if risk_value >= 50:
        send_notification(
            "⚠️ ALERTĂ METEO RoAlert",
            f"Risc de furtună: {risk_value}%! Verificați stația meteo pentru detalii."
        )
        new_data["alert_sent"] = True
    elif risk_value < 50:
        new_data["alert_sent"] = False
    else:
        new_data["alert_sent"] = latest_data.get("alert_sent", False)

    latest_data = new_data

    print("Date primite:", latest_data)
    return "OK", 200


@app.route('/alert.mp3')
def serve_alert():
    return send_from_directory(os.getcwd(), 'alert.mp3')


if __name__ == '__main__':
    print("Serverul RoAlert pornit. Ascultă la http://0.0.0.0:5000/")
    app.run(host='0.0.0.0', port=5000)