#include "mpu6050.h"
#include "config.h"

int16_t baseAccelX = 0, baseAccelY = 0, baseAccelZ = 0;
int16_t baseGyroX = 0, baseGyroY = 0, baseGyroZ = 0;
bool isCalibrated = false;

// Extern variables needed from main.cpp
extern hw_timer_t *sampleTimer;

bool mpu6050_init(void) {
  // Scan for I2C devices to ensure MPU6050 is connected
  scan_i2c_devices();
  
  // Reset MPU6050
  Serial.println("Resetting MPU6050...");
  for (int i = 0; i < 3; i++) {
    write_register(MPU6050_ADDR, MPU6050_PWR_MGMT_1, 0x80);
    delay(100);
  }
  
  // Sets the clock source to PLL with X-axis gyro as reference
  write_register(MPU6050_ADDR, MPU6050_PWR_MGMT_1, 0x01);
  delay(10);
  
  // Read WHO_AM_I register to verify connection
  uint8_t whoami = read_register(MPU6050_ADDR, MPU6050_WHO_AM_I);
  Serial.print("WHO_AM_I register: 0x");
  Serial.println(whoami, HEX);
  
  // Check WHO_AM_I value
  if (whoami != 0x68 && whoami != 0x72) {
    Serial.println("MPU6050 WHO_AM_I value incorect");
    
    // Check if it is a communication issue
    if (whoami == 0x83) {
      Serial.println("Posibilă problemă de comunicare I2C");
      // Try to reset I2C bus
      return false;
    }
    return false;
  }
  
  // Configure MPU6050 settings
  write_register(MPU6050_ADDR, MPU6050_SMPLRT_DIV, 0x07);  // Sample Rate Divider
  write_register(MPU6050_ADDR, MPU6050_CONFIG, 0x06);      // Filtru digital 5Hz
  write_register(MPU6050_ADDR, MPU6050_GYRO_CONFIG, 0x08); // Gyro ±500°/s
  write_register(MPU6050_ADDR, MPU6050_ACCEL_CONFIG, 0x08); // Accel ±4g
  
  // Activate all axes on the accelerometer and gyroscope
  write_register(MPU6050_ADDR, MPU6050_PWR_MGMT_2, 0x00);
  
  // Check if the configuration was successful
  uint8_t check = read_register(MPU6050_ADDR, MPU6050_GYRO_CONFIG);
  if (check != 0x08) {
    Serial.println("Failed to configure GYRO_CONFIG");
    return false;
  }
  
  // Test reading accelerometer and gyroscope data
  int16_t accelX, accelY, accelZ;
  int16_t gyroX, gyroY, gyroZ;
  
  // Try to read accelerometer data
  uint8_t buffer[6];
  bool readOk = true;
  
  try {
    read_registers(MPU6050_ADDR, MPU6050_ACCEL_XOUT_H, buffer, 6);
    accelX = (int16_t)((buffer[0] << 8) | buffer[1]);
    accelY = (int16_t)((buffer[2] << 8) | buffer[3]);
    accelZ = (int16_t)((buffer[4] << 8) | buffer[5]);
    
    Serial.println("Test accel readings:");
    Serial.println(accelX);
    Serial.println(accelY);
    Serial.println(accelZ);
  } catch(...) {
    readOk = false;
  }
  
  if (!readOk) {
    Serial.println("Failed to read accelerometer data");
    return false;
  }
  
  Serial.println("MPU6050 initialized successfully");
  return true;
}

void mpu6050_read_accel(int16_t *accelX, int16_t *accelY, int16_t *accelZ) {
  uint8_t buffer[6];
  
  // Read accelerometer data registers (0x3B-0x40)
  read_registers(MPU6050_ADDR, MPU6050_ACCEL_XOUT_H, buffer, 6);
  
  // Combine high and low bytes
  *accelX = (int16_t)((buffer[0] << 8) | buffer[1]);
  *accelY = (int16_t)((buffer[2] << 8) | buffer[3]);
  *accelZ = (int16_t)((buffer[4] << 8) | buffer[5]);
}

void mpu6050_read_gyro(int16_t *gyroX, int16_t *gyroY, int16_t *gyroZ) {
  uint8_t buffer[6];
  
  // Read gyroscope data registers (0x43-0x48)
  read_registers(MPU6050_ADDR, MPU6050_GYRO_XOUT_H, buffer, 6);
  
  // Combine high and low bytes
  *gyroX = (int16_t)((buffer[0] << 8) | buffer[1]);
  *gyroY = (int16_t)((buffer[2] << 8) | buffer[3]);
  *gyroZ = (int16_t)((buffer[4] << 8) | buffer[5]);
}

void calibrate_sensors(void) {
  Serial.println("=== CALIBRATING SENSORS ===");
  Serial.println("Keep the controller in NEUTRAL position!");
  
  // Disable timer during calibration
  if (sampleTimer != NULL) {
    timerAlarmDisable(sampleTimer);
  }
  
  // LED feedback during calibration
  for (int blink = 0; blink < 3; blink++) {
    digitalWrite(LED_BUILTIN, HIGH);
    delay(200);
    digitalWrite(LED_BUILTIN, LOW);
    delay(200);
  }
  
  int32_t sumAccelX = 0, sumAccelY = 0, sumAccelZ = 0;
  int32_t sumGyroX = 0, sumGyroY = 0, sumGyroZ = 0;
  const int numSamples = 100;
  
  Serial.println("Taking calibration samples...");
  for (int i = 0; i < numSamples; i++) {
    int16_t accelX, accelY, accelZ;
    int16_t gyroX, gyroY, gyroZ;
    
    mpu6050_read_accel(&accelX, &accelY, &accelZ);
    mpu6050_read_gyro(&gyroX, &gyroY, &gyroZ);
    
    sumAccelX += accelX;
    sumAccelY += accelY;
    sumAccelZ += accelZ;
    sumGyroX += gyroX;
    sumGyroY += gyroY;
    sumGyroZ += gyroZ;
    
    // Progress indicator
    if (i % 20 == 0) {
      digitalWrite(LED_BUILTIN, !digitalRead(LED_BUILTIN));
    }
    
    delay(10);
  }
  
  // Calculate NEUTRAL baseline values
  baseAccelX = sumAccelX / numSamples;
  baseAccelY = sumAccelY / numSamples;
  baseAccelZ = sumAccelZ / numSamples;
  baseGyroX = sumGyroX / numSamples;
  baseGyroY = sumGyroY / numSamples;
  baseGyroZ = sumGyroZ / numSamples;
  
  isCalibrated = true;
  
  Serial.println("=== CALIBRATION COMPLETE ===");
  Serial.printf("Neutral Accel: X=%d, Y=%d, Z=%d\n", baseAccelX, baseAccelY, baseAccelZ);
  Serial.printf("Neutral Gyro: X=%d, Y=%d, Z=%d\n", baseGyroX, baseGyroY, baseGyroZ);
  Serial.println("All movements now relative to this neutral position!");
  
  // Final LED confirmation
  for (int i = 0; i < 5; i++) {
    digitalWrite(LED_BUILTIN, HIGH);
    delay(100);
    digitalWrite(LED_BUILTIN, LOW);
    delay(100);
  }
  
  // Re-enable timer
  if (sampleTimer != NULL) {
    timerAlarmEnable(sampleTimer);
  }
}