/*
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * * * * * * * * * * * * * * * * ESP32 WIRELESS CONTROLLER * * * * * * * * * * * * *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 */

#include <Arduino.h>
#include "config.h"
#include "i2c_interface.h"
#include "ble_manager.h"
#include "mpu6050.h"
#include "controller.h"

// ESP32 timer handler for sensor sampling
hw_timer_t *sampleTimer = NULL;
portMUX_TYPE timerMux = portMUX_INITIALIZER_UNLOCKED;
portMUX_TYPE buttonMux = portMUX_INITIALIZER_UNLOCKED;
volatile bool readSensorFlag = false;
volatile bool calibrateButtonPressed = false;
volatile unsigned long lastCalibrateButtonTime = 0;
portMUX_TYPE calibrateButtonMux = portMUX_INITIALIZER_UNLOCKED;

// Timer interrupt handler
void IRAM_ATTR onSampleTimer() {
  portENTER_CRITICAL_ISR(&timerMux);
  readSensorFlag = true;
  portEXIT_CRITICAL_ISR(&timerMux);
}

// Button interrupt handler
void IRAM_ATTR buttonISR() {
  portENTER_CRITICAL_ISR(&buttonMux);
  unsigned long currentTime = millis();
  
  // Debounce - only process if enough time has passed
  if (currentTime - lastButtonInterruptTime > DEBOUNCE_TIME_MS) {
    lastButtonInterruptTime = currentTime;
    buttonInterruptOccurred = true;
  }
  portEXIT_CRITICAL_ISR(&buttonMux);
}

// Button calibration interrupt handler
void IRAM_ATTR calibrateButtonISR() {
  portENTER_CRITICAL_ISR(&calibrateButtonMux);
  unsigned long currentTime = millis();
  
  // Debounce - only process if enough time has passed
  if (currentTime - lastCalibrateButtonTime > DEBOUNCE_TIME_MS) {
    lastCalibrateButtonTime = currentTime;
    calibrateButtonPressed = true;
  }
  portEXIT_CRITICAL_ISR(&calibrateButtonMux);
}

void setup() {
  // Initialize serial communication for debugging
  Serial.begin(SERIAL_BAUD_RATE);
  Serial.println("Wireless Glove Controller initializing...");
  
  // Offers a short delay to allow serial monitor to connect
  delay(300);
  
  // Set pull-up resistors for I2C pins
  pinMode(SDA_PIN, INPUT_PULLUP);
  pinMode(SCL_PIN, INPUT_PULLUP);
  delay(50);
  
  // Continue with normal pin setup
  pinMode(MODE_BUTTON_PIN, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(MODE_BUTTON_PIN), buttonISR, FALLING);

  // Set up calibration button
  pinMode(CALIBRATE_BUTTON_PIN, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(CALIBRATE_BUTTON_PIN), calibrateButtonISR, FALLING);
  
  // Initialize I2C interface
  i2c_init();
  
  // Offers a short delay to ensure I2C is ready
  delay(100);
  
  // Verifies if the MPU6050 is connected and initializes it
  if (!mpu6050_init()) {
    Serial.println("MPU6050 initialization failed");
    while(1) {
      digitalWrite(LED_BUILTIN, !digitalRead(LED_BUILTIN));  // Blinking error indicator
      delay(300);
    }
  }

  // Inițializare BLE
  initBLE();
  
  // Wait for sensor to stabilize, then calibrate
  delay(1000);
  calibrate_sensors();
  
  // Set up sample timer (for sensor reading)
  sampleTimer = timerBegin(0, 80, true); // 80 divider for 1 MHz
  timerAttachInterrupt(sampleTimer, &onSampleTimer, true);
  timerAlarmWrite(sampleTimer, SAMPLE_RATE_MS * 1000, true); // Convert ms to µs
  timerAlarmEnable(sampleTimer);
  
  Serial.println("System initialized. Current mode: Mouse Control");
}

void loop() {
  // Process sensor data when timer flag is set
  if (readSensorFlag) {
    portENTER_CRITICAL(&timerMux);
    readSensorFlag = false;
    portEXIT_CRITICAL(&timerMux);
    
    // Process if device is connected
    if (deviceConnected) {
      process_motion_data();
    }
  }
  
  // Process button interrupt
  if (buttonInterruptOccurred) {
    portENTER_CRITICAL(&buttonMux);
    buttonInterruptOccurred = false;
    portEXIT_CRITICAL(&buttonMux);
    
    handle_button_interrupt();
  }

  // Process calibration button press
  if (calibrateButtonPressed) {
    portENTER_CRITICAL(&calibrateButtonMux);
    calibrateButtonPressed = false;
    portEXIT_CRITICAL(&calibrateButtonMux);
    
    // Check if the button was pressed long enough
    if (millis() - lastCalibrateButtonTime > DEBOUNCE_TIME_MS) {
      Serial.println("Calibration button pressed, recalibrating sensors...");
      calibrate_sensors();
    }
  }
  
  // Gestionare reconectare BLE
  handleBLEEvents();
  
  // Small delay to prevent CPU hogging
  delay(1);
}

// Handles calibration button press
void handle_calibrate_button() {
  Serial.println("Calibrate button pressed - starting calibration");
  calibrate_sensors();
  
  // Trimite comandă pentru centrarea cursorului
  send_center_cursor_command();
  
  // LED feedback
  digitalWrite(LED_BUILTIN, HIGH);
  delay(200);
  digitalWrite(LED_BUILTIN, LOW);
}