#include "i2c_interface.h"

void i2c_init(void) {
  // Set up I2C pins
  pinMode(SDA_PIN, OUTPUT);
  pinMode(SCL_PIN, OUTPUT);
  
  // Set both lines high
  digitalWrite(SDA_PIN, HIGH);
  digitalWrite(SCL_PIN, HIGH);
  
  Serial.println("I2C initialized");
}

void i2c_start(void) {
  // START condition: SDA goes from HIGH to LOW while SCL is HIGH
  digitalWrite(SDA_PIN, HIGH);
  digitalWrite(SCL_PIN, HIGH);
  delayMicroseconds(4);
  
  digitalWrite(SDA_PIN, LOW);
  delayMicroseconds(4);
  digitalWrite(SCL_PIN, LOW);
}

void i2c_stop(void) {
  // STOP condition: SDA goes from LOW to HIGH while SCL is HIGH
  digitalWrite(SDA_PIN, LOW);
  delayMicroseconds(4);
  
  digitalWrite(SCL_PIN, HIGH);
  delayMicroseconds(4);
  
  digitalWrite(SDA_PIN, HIGH);
  delayMicroseconds(4);
}

bool i2c_write(uint8_t data) {
  // Send 8 bits, MSB first
  for (int i = 7; i >= 0; i--) {
    // Set SDA before SCL rises
    if (data & (1 << i)) {
      digitalWrite(SDA_PIN, HIGH);
    } else {
      digitalWrite(SDA_PIN, LOW);
    }
    
    delayMicroseconds(5);
    digitalWrite(SCL_PIN, HIGH);
    delayMicroseconds(10);
    digitalWrite(SCL_PIN, LOW);
    delayMicroseconds(5);
  }
  
  // 9th clock cycle for ACK
  // Release SDA for slave to control (pull down for ACK)
  pinMode(SDA_PIN, INPUT_PULLUP);
  
  digitalWrite(SCL_PIN, HIGH);
  delayMicroseconds(10);
  
  // Read ACK (LOW = ACK, HIGH = NACK)
  bool ack = (digitalRead(SDA_PIN) == LOW);
  
  digitalWrite(SCL_PIN, LOW);
  delayMicroseconds(5);
  
  // Regain control of SDA line
  pinMode(SDA_PIN, OUTPUT);
  
  if (!ack) {
    Serial.println("I2C NACK received");
    return false;
  }
  
  return true;
}

void i2c_read_ack(uint8_t *data) {
  uint8_t result = 0;
  
  // Release SDA for slave to control
  pinMode(SDA_PIN, INPUT_PULLUP);
  
  // Read 8 bits, MSB first
  for (int i = 7; i >= 0; i--) {
    digitalWrite(SCL_PIN, HIGH);
    delayMicroseconds(4);
    
    if (digitalRead(SDA_PIN)) {
      result |= (1 << i);
    }
    
    digitalWrite(SCL_PIN, LOW);
    delayMicroseconds(4);
  }
  
  // Send ACK (pull SDA LOW during 9th clock)
  pinMode(SDA_PIN, OUTPUT);
  digitalWrite(SDA_PIN, LOW);
  
  digitalWrite(SCL_PIN, HIGH);
  delayMicroseconds(4);
  digitalWrite(SCL_PIN, LOW);
  delayMicroseconds(4);
  
  *data = result;
}

void i2c_read_nack(uint8_t *data) {
  uint8_t result = 0;
  
  // Release SDA for slave to control
  pinMode(SDA_PIN, INPUT_PULLUP);
  
  // Read 8 bits, MSB first
  for (int i = 7; i >= 0; i--) {
    digitalWrite(SCL_PIN, HIGH);
    delayMicroseconds(4);
    
    if (digitalRead(SDA_PIN)) {
      result |= (1 << i);
    }
    
    digitalWrite(SCL_PIN, LOW);
    delayMicroseconds(4);
  }
  
  // Send NACK (keep SDA HIGH during 9th clock)
  pinMode(SDA_PIN, OUTPUT);
  digitalWrite(SDA_PIN, HIGH);
  
  digitalWrite(SCL_PIN, HIGH);
  delayMicroseconds(4);
  digitalWrite(SCL_PIN, LOW);
  delayMicroseconds(4);
  
  *data = result;
}

bool write_register(uint8_t device_addr, uint8_t register_addr, uint8_t value) {
  i2c_start();
  
  // Write device address with WRITE flag
  i2c_write(device_addr << 1);
  
  // Write register address
  i2c_write(register_addr);
  
  // Write value
  i2c_write(value);
  
  i2c_stop();
  
  return true;
}

uint8_t read_register(uint8_t device_addr, uint8_t register_addr) {
  uint8_t result;
  
  i2c_start();
  
  // Write device address with WRITE flag
  i2c_write(device_addr << 1);
  
  // Write register address
  i2c_write(register_addr);
  
  // Restart
  i2c_start();
  
  // Write device address with READ flag
  i2c_write((device_addr << 1) | 0x01);
  
  // Read data
  i2c_read_nack(&result);
  
  i2c_stop();
  
  return result;
}

void read_registers(uint8_t device_addr, uint8_t start_register, uint8_t *buffer, uint8_t length) {
  i2c_start();
  
  // Write device address with WRITE flag
  i2c_write(device_addr << 1);
  
  // Write starting register address
  i2c_write(start_register);
  
  // Restart
  i2c_start();
  
  // Write device address with READ flag
  i2c_write((device_addr << 1) | 0x01);
  
  // Read data bytes with ACK except the last one
  for (uint8_t i = 0; i < length - 1; i++) {
    i2c_read_ack(&buffer[i]);
  }
  
  // Read last byte with NACK
  i2c_read_nack(&buffer[length - 1]);
  
  i2c_stop();
}

void scan_i2c_devices() {
  Serial.println("Scanning I2C bus...");
  
  for (uint8_t addr = 0x00; addr <= 0x7F; addr++) {
    i2c_start();
    
    // Write device address with WRITE flag
    i2c_write(addr << 1);
    
    // Check if ACK was received
    pinMode(SDA_PIN, INPUT_PULLUP);
    digitalWrite(SCL_PIN, HIGH);
    delayMicroseconds(5);
    
    bool ack = (digitalRead(SDA_PIN) == LOW);
    
    digitalWrite(SCL_PIN, LOW);
    delayMicroseconds(5);
    pinMode(SDA_PIN, OUTPUT);
    
    i2c_stop();
    
    if (ack) {
      Serial.print("Found I2C device at address 0x");
      Serial.println(addr, HEX);
    }
  }
  
  Serial.println("Scan complete.");
}