#include "controller.h"
#include "ble_manager.h"

ControlMode currentMode = MOUSE_MODE;
volatile bool buttonInterruptOccurred = false;
volatile unsigned long lastButtonInterruptTime = 0;
float mouseAcceleration = 1.5;
float keyboardSensitivity = 1.0;

void send_center_cursor_command() {
  if (!deviceConnected) return;
  
  uint8_t buffer[4];
  buffer[0] = CENTER_CURSOR;
  buffer[1] = 0;
  buffer[2] = 0;
  buffer[3] = 0;
  
  pTxCharacteristic->setValue(buffer, 4);
  pTxCharacteristic->notify();
  
  Serial.println("Center cursor command sent");
}

void process_motion_data(void) {
  if (!isCalibrated) {
    return;
  }
  
  int16_t accelX, accelY, accelZ;
  int16_t gyroX, gyroY, gyroZ;
  
  mpu6050_read_accel(&accelX, &accelY, &accelZ);
  mpu6050_read_gyro(&gyroX, &gyroY, &gyroZ);
  
  // Calculate change from calibrated baseline
  int16_t deltaAccelX = accelX - baseAccelX;
  int16_t deltaAccelY = accelY - baseAccelY;
  int16_t deltaGyroX = gyroX - baseGyroX;
  int16_t deltaGyroY = gyroY - baseGyroY;
  
  if (currentMode == MOUSE_MODE) {
    // Mouse control - movement on tilting hand
    int16_t processedGyroX = -deltaGyroY;
    int16_t processedGyroY = deltaGyroX;
    
    // Apply deadzone to gyro values
    if (abs(processedGyroX) < GYRO_DEADZONE) processedGyroX = 0;
    if (abs(processedGyroY) < GYRO_DEADZONE) processedGyroY = 0;
    
    // Generate movement if there are significant values relative to baseline
    if (processedGyroX != 0 || processedGyroY != 0) {
      // Calculate non-linear scaling relative to calibrated position
      float scaledX = constrain((float)processedGyroX / GYRO_SCALE_MAX, -1.0f, 1.0f);
      float scaledY = constrain((float)processedGyroY / GYRO_SCALE_MAX, -1.0f, 1.0f);

      // Apply curved scaling for smoother movement
      float curvedX = scaledX * abs(scaledX) * mouseAcceleration * MOUSE_SPEED_MAX;
      float curvedY = scaledY * abs(scaledY) * mouseAcceleration * MOUSE_SPEED_MAX;
      
      int16_t mouseX = curvedX;
      int16_t mouseY = curvedY;
      
      send_mouse_command(mouseX, mouseY);
    }
  } else {  // KEYBOARD_MODE
    // Keyboard control - tine tastele apăsate continuu cât timp mâna e înclinată
    static bool w_pressed = false, a_pressed = false, s_pressed = false, d_pressed = false;
    
    // Using thresholds to determine key presses
    bool should_press_w = deltaAccelY < ACCEL_THRESHOLD;
    bool should_press_s = deltaAccelY > -ACCEL_THRESHOLD;
    bool should_press_a = deltaAccelX > ACCEL_THRESHOLD;
    bool should_press_d = deltaAccelX < -ACCEL_THRESHOLD;
    
    // Forward/backward (W/S) - continuous press
    if (should_press_w && !w_pressed) {
      send_keyboard_command('w', true);
      w_pressed = true;
      Serial.println("W pressed (continuous)");
    } else if (!should_press_w && w_pressed) {
      send_keyboard_command('w', false);
      w_pressed = false;
      Serial.println("W released");
    }
    
    if (should_press_s && !s_pressed) {
      send_keyboard_command('s', true);
      s_pressed = true;
      Serial.println("S pressed (continuous)");
    } else if (!should_press_s && s_pressed) {
      send_keyboard_command('s', false);
      s_pressed = false;
      Serial.println("S released");
    }
    
    // Left/right (A/D) - continuous press
    if (should_press_a && !a_pressed) {
      send_keyboard_command('a', true);
      a_pressed = true;
      Serial.println("A pressed (continuous)");
    } else if (!should_press_a && a_pressed) {
      send_keyboard_command('a', false);
      a_pressed = false;
      Serial.println("A released");
    }
    
    if (should_press_d && !d_pressed) {
      send_keyboard_command('d', true);
      d_pressed = true;
      Serial.println("D pressed (continuous)");
    } else if (!should_press_d && d_pressed) {
      send_keyboard_command('d', false);
      d_pressed = false;
      Serial.println("D released");
    }
  }
}

void handle_button_interrupt(void) {
  
  // Button was pressed - switch modes
  switch_mode();
  
  Serial.println("Button interrupt handled");
}


void switch_mode(void) {
  if (currentMode == MOUSE_MODE) {
    currentMode = KEYBOARD_MODE;
    Serial.println("Switched to Keyboard Control Mode (via BLE command)");
  } else {
    currentMode = MOUSE_MODE;
    Serial.println("Switched to Mouse Control Mode (via BLE command)");
  }
  
  calibrate_sensors();
  
  // Send confirmation of mode change
  send_status_update();
}

void send_mouse_command(int16_t x, int16_t y) {
  // Verify if device is connected
  if (!deviceConnected) {
    return;
  }
  
  // Constrain movement values
  if (x > 127) x = 127;
  if (x < -127) x = -127;
  if (y > 127) y = 127;
  if (y < -127) y = -127;
  
  // Prepare command packet
  uint8_t buffer[4];
  buffer[0] = MOUSE_MOVE;  // Command code for mouse movement
  
  // Asigură-te că valorile negative sunt codificate corect
  buffer[1] = (uint8_t)(x & 0xFF);
  buffer[2] = (uint8_t)(y & 0xFF);
  buffer[3] = 0;
  
  // Trimite prin BLE
  pTxCharacteristic->setValue(buffer, 4);
  pTxCharacteristic->notify();
  
  // Debug output (limited to avoid flooding serial)
  static unsigned long lastDebugTime = 0;
  unsigned long currentTime = millis();
  if (currentTime - lastDebugTime > 500) {
    Serial.print("Mouse move: X=");
    Serial.print(x);
    Serial.print(", Y=");
    Serial.println(y);
    lastDebugTime = currentTime;
  }
}

void send_keyboard_command(char key, bool pressed) {
  // Verify if device is connected
  if (!deviceConnected) {
    return;
  }
  
  // Prepare command packet
  uint8_t buffer[3];
  buffer[0] = pressed ? KEY_PRESS : KEY_RELEASE;
  buffer[1] = key;
  buffer[2] = 0;  // Reserved
  
  // Send through BLE
  pTxCharacteristic->setValue(buffer, 3);
  pTxCharacteristic->notify();
  
  Serial.print("Key ");
  Serial.print(key);
  Serial.println(pressed ? " pressed" : " released");
}