#include "ble_manager.h"
#include "controller.h"

BLEServer *pServer = NULL;
BLECharacteristic *pTxCharacteristic = NULL;
bool deviceConnected = false;
bool oldDeviceConnected = false;

// BLE callbacks
void MyServerCallbacks::onConnect(BLEServer* pServer) {
  deviceConnected = true;
  Serial.println("Device connected");
}

void MyServerCallbacks::onDisconnect(BLEServer* pServer) {
  deviceConnected = false;
  Serial.println("Device disconnected");
}

void MyCallbacks::onWrite(BLECharacteristic *pCharacteristic) {
  std::string rxValue = pCharacteristic->getValue();

  if (rxValue.length() > 0) {
    uint8_t cmd = rxValue[0];
    
    Serial.print("Received command: ");
    Serial.println(cmd);
    
    switch(cmd) {
      case CHANGE_MODE:
        // Schimbă modul de funcționare
        switch_mode();
        send_status_update();
        break;
        
      case SET_SENSITIVITY:
        if (rxValue.length() >= 3) {
          float newSensitivity = (float)rxValue[1] / 10.0f;  // Scalare de la 0-100 la 0.0-10.0
          if (currentMode == MOUSE_MODE) {
            mouseAcceleration = newSensitivity;
          } else {
            keyboardSensitivity = newSensitivity;
          }
          Serial.print("Sensitivity updated to: ");
          Serial.println(newSensitivity);
        }
        break;
        
      case CALIBRATE:
        // Declanșează calibrarea
        calibrate_sensors();
        send_status_update();
        break;
        
      case GET_STATUS:
        // Trimite starea curentă
        send_status_update();
        break;
        
      default:
        Serial.println("Unknown command received");
        break;
    }
  }
}

void send_status_update() {
  if (!deviceConnected) return;
  
  uint8_t buffer[4];
  buffer[0] = GET_STATUS;  // Cod pentru răspuns la status
  buffer[1] = currentMode; // 0 = MOUSE_MODE, 1 = KEYBOARD_MODE
  buffer[2] = (uint8_t)(mouseAcceleration * 10);     // Sensibilitate mouse (0-100)
  buffer[3] = (uint8_t)(keyboardSensitivity * 10);   // Sensibilitate keyboard (0-100)
  
  pTxCharacteristic->setValue(buffer, 4);
  pTxCharacteristic->notify();
}

void initBLE() {
  // Inițializare BLE
  BLEDevice::init(BT_DEVICE_NAME);
  
  // Creează server BLE
  pServer = BLEDevice::createServer();
  pServer->setCallbacks(new MyServerCallbacks());
  
  // Creează serviciu BLE
  BLEService *pService = pServer->createService(SERVICE_UUID);
  
  // Creează caracteristică pentru transmisie (spre PC)
  pTxCharacteristic = pService->createCharacteristic(
                        CHARACTERISTIC_UUID_TX,
                        BLECharacteristic::PROPERTY_NOTIFY
                      );
  pTxCharacteristic->addDescriptor(new BLE2902());
  
  // Creează caracteristică pentru recepție (de la PC) - opțional
  BLECharacteristic *pRxCharacteristic = pService->createCharacteristic(
                                           CHARACTERISTIC_UUID_RX,
                                           BLECharacteristic::PROPERTY_WRITE
                                         );
  pRxCharacteristic->setCallbacks(new MyCallbacks());
  
  // Pornește serviciul
  pService->start();
  
  // Începe advertising (face dispozitivul vizibil)
  BLEAdvertising *pAdvertising = BLEDevice::getAdvertising();
  pAdvertising->addServiceUUID(SERVICE_UUID);
  pAdvertising->setScanResponse(true);
  pAdvertising->setMinPreferred(0x06);  // Funcții iPhone
  pAdvertising->setMinPreferred(0x12);
  BLEDevice::startAdvertising();
  Serial.println("BLE started, device is now discoverable as: " + String(BT_DEVICE_NAME));
}

void handleBLEEvents() {
  // Gestionare reconectare BLE
  if (!deviceConnected && oldDeviceConnected) {
    delay(500); // Dă timp stivei BLE să se pregătească
    pServer->startAdvertising(); // Reîncepe advertising
    Serial.println("Start advertising");
    oldDeviceConnected = deviceConnected;
  }
  
  // Conectare nouă
  if (deviceConnected && !oldDeviceConnected) {
    oldDeviceConnected = deviceConnected;
  }
}