"""
ESP32 Wireless Controller Receptor cu Bleak
---------------------------------
Acest program folosește biblioteca Bleak pentru comunicarea Bluetooth
"""

import tkinter as tk
from tkinter import ttk, messagebox, Scale
import asyncio
import threading
import struct
import time
from pynput.mouse import Button, Controller as MouseController
from pynput.keyboard import Key, Controller as KeyboardController


try:
    from bleak import BleakScanner, BleakClient
    has_bleak = True
except ImportError:
    has_bleak = False
    messagebox.showerror("Eroare", "Biblioteca 'bleak' nu este instalată. Instalează-o folosind: pip install bleak")
    exit(1)

# Command codes for ESP32 controller
MOUSE_MOVE = 1
KEY_PRESS = 2
KEY_RELEASE = 3
CENTER_CURSOR = 4
CHANGE_MODE = 5
SET_SENSITIVITY = 6
CALIBRATE = 7
GET_STATUS = 8

# Service and characteristic UUIDs for UART communication
UART_SERVICE_UUID = "6E400001-B5A3-F393-E0A9-E50E24DCCA9E"
UART_RX_CHAR_UUID = "6E400002-B5A3-F393-E0A9-E50E24DCCA9E"
UART_TX_CHAR_UUID = "6E400003-B5A3-F393-E0A9-E50E24DCCA9E"

class AsyncThread:
    """Clasă pentru a rula funcții async în thread-uri"""
    def __init__(self):
        self.loop = asyncio.new_event_loop()
        self.thread = threading.Thread(target=self._run_loop, daemon=True)
        self.thread.start()
        
    def _run_loop(self):
        asyncio.set_event_loop(self.loop)
        self.loop.run_forever()
        
    def run_async(self, coro):
        return asyncio.run_coroutine_threadsafe(coro, self.loop)
        
    def stop(self):
        self.loop.call_soon_threadsafe(self.loop.stop)
        self.thread.join(timeout=1.0)


class GloveControllerBLEApp:
    def __init__(self, root):
        self.root = root
        self.root.title("ESP32 Wireless Controller - BLE")
        self.root.geometry("450x550")
        self.root.resizable(False, False)

        self.screen_width = root.winfo_screenwidth()
        self.screen_height = root.winfo_screenheight()

        self.current_mode = 0  # 0 = Mouse, 1 = Keyboard
        self.mouse_sensitivity = 50
        self.keyboard_sensitivity = 50

        # Tracking commands variables
        self.rx_characteristic = None
        
        # Mouse/Keyboard controllers
        self.mouse = MouseController()
        self.keyboard = KeyboardController()
        
        # BLE comunication variables
        self.ble_client = None
        self.connected = False
        self.running = False
        self.receive_buffer = bytearray()
        
        # Thread async for BLE operations
        self.async_thread = AsyncThread()
        
        # Graphic interface
        self.create_gui()
        
        # Initial scan for devices
        self.scan_devices()
        
    def create_gui(self):
        # Styles
        style = ttk.Style()
        style.configure("TFrame", background="#f0f0f0")
        style.configure("Header.TLabel", font=("Arial", 14, "bold"), background="#f0f0f0")
        style.configure("TLabel", background="#f0f0f0")
        style.configure("TButton", font=("Arial", 10))
        
        # Main canvas and scrollbar
        self.canvas = tk.Canvas(self.root, bg="#f0f0f0")
        self.canvas.pack(side=tk.LEFT, fill=tk.BOTH, expand=True)
        
        # Vertical scrollbar for the canvas
        main_scrollbar = ttk.Scrollbar(self.root, orient="vertical", command=self.canvas.yview)
        main_scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        
        # Configure canvas to use scrollbar
        self.canvas.configure(yscrollcommand=main_scrollbar.set)
        
        # Scrollable frame inside the canvas
        self.scrollable_frame = ttk.Frame(self.canvas)
        self.canvas_window = self.canvas.create_window((0, 0), window=self.scrollable_frame, anchor="nw")
        
        # Functions to configure scroll region and canvas width
        def configure_scroll_region(event=None):
            self.canvas.configure(scrollregion=self.canvas.bbox("all"))
        
        def configure_canvas_width(event=None):
            canvas_width = self.canvas.winfo_width()
            self.canvas.itemconfig(self.canvas_window, width=canvas_width)
        
        # Mouse wheel scroll
        def on_mousewheel(event):
            self.canvas.yview_scroll(int(-1*(event.delta/120)), "units")
        
        # Bind events
        self.scrollable_frame.bind("<Configure>", configure_scroll_region)
        self.canvas.bind("<Configure>", configure_canvas_width)
        self.canvas.bind("<MouseWheel>", on_mousewheel)
        
        # Bind mouse wheel to the canvas when mouse enters
        def bind_to_mousewheel(event):
            self.canvas.bind_all("<MouseWheel>", on_mousewheel)
        
        def unbind_from_mousewheel(event):
            self.canvas.unbind_all("<MouseWheel>")
        
        self.canvas.bind('<Enter>', bind_to_mousewheel)
        self.canvas.bind('<Leave>', unbind_from_mousewheel)
        
        main_frame = ttk.Frame(self.scrollable_frame, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)
        
        # Application title
        header_frame = ttk.Frame(main_frame)
        header_frame.pack(fill=tk.X, pady=(0, 10))
        
        title_label = ttk.Label(header_frame, text="ESP32 Wireless Controller (BLE)", style="Header.TLabel")
        title_label.pack(side=tk.LEFT)
        
        # Status of connection
        status_frame = ttk.LabelFrame(main_frame, text="Stare conexiune", padding="10")
        status_frame.pack(fill=tk.X, pady=(0, 10))
        
        ttk.Label(status_frame, text="Status:").pack(side=tk.LEFT)
        self.status_label = ttk.Label(status_frame, text="Deconectat", foreground="red")
        self.status_label.pack(side=tk.LEFT, padx=5)
        
        self.mode_label = ttk.Label(status_frame, text="Mod: -")
        self.mode_label.pack(side=tk.RIGHT)
        
        # Device selection frame
        device_frame = ttk.LabelFrame(main_frame, text="Dispozitiv Bluetooth LE", padding="10")
        device_frame.pack(fill=tk.X, pady=(0, 10))
        
        self.device_combo = ttk.Combobox(device_frame, state="readonly", width=40)
        self.device_combo.pack(side=tk.TOP, fill=tk.X, padx=5, pady=5)
        
        btn_frame = ttk.Frame(device_frame)
        btn_frame.pack(fill=tk.X)
        
        scan_btn = ttk.Button(btn_frame, text="Caută dispozitive", command=self.scan_devices)
        scan_btn.pack(side=tk.LEFT, padx=5)
        
        services_btn = ttk.Button(btn_frame, text="Verifică servicii", command=self.check_services)
        services_btn.pack(side=tk.LEFT, padx=5)
        
        self.connect_btn = ttk.Button(btn_frame, text="Conectare", command=self.toggle_connection)
        self.connect_btn.pack(side=tk.RIGHT, padx=5)

        # Control frame
        control_frame = ttk.LabelFrame(main_frame, text="Control Mod", padding="10")
        control_frame.pack(fill=tk.X, pady=(0, 10))
        
        # Control buttons
        control_row1 = tk.Frame(control_frame)
        control_row1.pack(fill=tk.X, pady=2)
        
        self.mode_button = tk.Button(control_row1, text="Schimbă în Mod Keyboard", 
                                    command=self.toggle_mode, bg="lightblue")
        self.mode_button.pack(side=tk.LEFT, padx=5)
        
        calibrate_button = tk.Button(control_row1, text="Calibrează Senzori", 
                                    command=self.calibrate_sensors, bg="orange")
        calibrate_button.pack(side=tk.LEFT, padx=5)
        
        # Extra control buttons
        control_row2 = tk.Frame(control_frame)
        control_row2.pack(fill=tk.X, pady=2)
        
        center_button = tk.Button(control_row2, text="Centrează Cursor", 
                                 command=self.center_cursor, bg="lightcoral")
        center_button.pack(side=tk.LEFT, padx=5)
        
        status_button = tk.Button(control_row2, text="Verifică Status", 
                                 command=self.request_status, bg="lightgreen")
        status_button.pack(side=tk.LEFT, padx=5)
        
        # Sensitivity control
        sensitivity_frame = ttk.LabelFrame(main_frame, text="Sensibilitate", padding="10")
        sensitivity_frame.pack(fill=tk.X, pady=(0, 10))
        
        # Sensitivity scale
        tk.Label(sensitivity_frame, text="Sensibilitate:").pack(side=tk.LEFT)
        self.sensitivity_scale = Scale(sensitivity_frame, from_=1, to=100, 
                                    orient=tk.HORIZONTAL, command=self.on_sensitivity_change)
        self.sensitivity_scale.set(50)
        self.sensitivity_scale.pack(side=tk.LEFT, fill=tk.X, expand=True, padx=10)

        # Log frame
        log_frame = ttk.LabelFrame(main_frame, text="Monitor activitate", padding="10")
        log_frame.pack(fill=tk.X, pady=(0, 10))  # Schimbat din fill=tk.BOTH, expand=True
        
        self.log_text = tk.Text(log_frame, height=8, wrap=tk.WORD)  # Redus de la 10 la 8
        self.log_text.pack(side=tk.LEFT, fill=tk.X, expand=True)  # Schimbat fill
        
        scrollbar = ttk.Scrollbar(log_frame, orient="vertical", command=self.log_text.yview)
        scrollbar.pack(side=tk.RIGHT, fill=tk.Y)
        self.log_text.config(yscrollcommand=scrollbar.set)
        self.log_text.config(state=tk.DISABLED)
        
        # Test commands frame
        test_frame = ttk.LabelFrame(main_frame, text="Test comenzi", padding="10")
        test_frame.pack(fill=tk.X, pady=(0, 10))
        
        ttk.Button(test_frame, text="Test Mișcare Mouse", command=self.test_mouse_move).pack(side=tk.LEFT, padx=5)
        ttk.Button(test_frame, text="Test Tastatură W", command=lambda: self.test_keyboard('w')).pack(side=tk.RIGHT, padx=5)
        
        # Information frame
        info_frame = ttk.Frame(main_frame)
        info_frame.pack(fill=tk.X, pady=(0, 10))
        
        info_text = "ESP32 Wireless Controller - BLE Version\n"
        info_text += "Mod Mouse: Controlează cursorul prin mișcarea mâinii\n"
        info_text += "Mod Tastatură: Controlează WASD prin înclinarea mâinii\n"
        info_text += "Bibliotecă: Bleak pentru comunicare Bluetooth LE"
        
        info_label = ttk.Label(info_frame, text=info_text, justify=tk.LEFT, wraplength=430)
        info_label.pack(pady=5)

        
        
    def log(self, message):
        """Adaugă mesaj în log"""
        self.log_text.config(state=tk.NORMAL)
        self.log_text.insert(tk.END, f"{time.strftime('%H:%M:%S')} - {message}\n")
        self.log_text.see(tk.END)
        self.log_text.config(state=tk.DISABLED)
    
    async def _scan_for_devices(self):
        """Scanare pentru dispozitive BLE - rulează async"""
        self.log("Scanare dispozitive Bluetooth LE...")
        devices = await BleakScanner.discover()
        
        if not devices:
            return []
        
        result = []
        for device in devices:
            if device.name:
                result.append((device.name, device.address))
                
        return result
    
    def scan_devices(self):
        """Interfață pentru scanare dispozitive"""
        async def _scan_and_update():
            devices = await self._scan_for_devices()
            
            if not devices:
                self.log("Nu s-au găsit dispozitive BLE")
                self.device_combo["values"] = ["Nu s-au găsit dispozitive"]
                self.device_combo.current(0)
                return
            
            device_strings = []
            self.device_map = {}
            
            for name, address in devices:
                device_str = f"{name} ({address})"
                device_strings.append(device_str)
                self.device_map[device_str] = (name, address)
                self.log(f"Găsit dispozitiv: {name} la adresa {address}")
            
            self.device_combo["values"] = device_strings
            self.device_combo.current(0)
            self.log(f"S-au găsit {len(devices)} dispozitive BLE")
            
        self.async_thread.run_async(_scan_and_update())
    
    def check_services(self):
        """Verifică serviciile disponibile pe dispozitivul selectat"""
        if not hasattr(self, "device_map") or not self.device_combo.get() in self.device_map:
            messagebox.showerror("Eroare", "Selectați un dispozitiv valid")
            return
        
        name, address = self.device_map[self.device_combo.get()]
        self.log(f"Verificare servicii pentru {name}...")
        
        async def _check_services():
            try:
                async with BleakClient(address) as client:
                    services = client.services
                    
                    for service in services:
                        self.log(f"Serviciu: {service.uuid}")
                        
                        for char in service.characteristics:
                            props = []
                            if "read" in char.properties:
                                props.append("read")
                            if "write" in char.properties:
                                props.append("write")
                            if "notify" in char.properties:
                                props.append("notify")
                                
                            self.log(f"  Caracteristică: {char.uuid}")
                            self.log(f"    Proprietăți: {', '.join(props)}")
                            
                            if "notify" in char.properties and UART_TX_CHAR_UUID.lower() in char.uuid.lower():
                                self.log(f"    *** Aceasta este caracteristica pentru recepție date ***")
                                
                            if "write" in char.properties and UART_RX_CHAR_UUID.lower() in char.uuid.lower():
                                self.log(f"    *** Aceasta este caracteristica pentru transmisie date ***")
            
            except Exception as e:
                self.log(f"Eroare la verificarea serviciilor: {e}")
                
        self.async_thread.run_async(_check_services())
    
    def toggle_connection(self):
        """Conectare/Deconectare la dispozitivul selectat"""
        if not self.connected:
            self.connect()
        else:
            self.disconnect()
    
    def connect(self):
        """Conectare la dispozitivul BLE selectat"""
        if not hasattr(self, "device_map") or not self.device_combo.get() in self.device_map:
            messagebox.showerror("Eroare", "Selectați un dispozitiv valid")
            return
            
        name, address = self.device_map[self.device_combo.get()]
        self.log(f"Conectare la {name}...")
        
        async def _connect():
            try:
                # Connect to the BLE device
                client = BleakClient(address)
                await client.connect()
                
                if not client.is_connected:
                    self.log("Conexiunea a eșuat")
                    return
                
                self.ble_client = client
                self.connected = True
                self.running = True
                
                # UI updates
                self.root.after(0, lambda: self.connect_btn.config(text="Deconectare"))
                self.root.after(0, lambda: self.status_label.config(text="Conectat", foreground="green"))
                self.log(f"Conectat cu succes la {name}")
                
                # Locate the characteristic for sending commands
                tx_char = None
                for service in client.services:
                    for char in service.characteristics:
                        if "notify" in char.properties and UART_TX_CHAR_UUID.lower() in char.uuid.lower():
                            tx_char = char
                            break
                
                if not tx_char:
                    self.log("Nu s-a găsit caracteristica pentru notificări")
                    await client.disconnect()
                    self.connected = False
                    self.running = False
                    return
                
                # Set a notification handler
                await client.start_notify(tx_char.uuid, self.notification_handler)
                
                # Wait for connection closure
                while self.running:
                    await asyncio.sleep(0.1)
                
                # Disconnect when done
                await client.disconnect()
                
            except Exception as e:
                self.log(f"Eroare la conectare: {str(e)}")
                self.connected = False
                self.running = False
                self.root.after(0, lambda: self.connect_btn.config(text="Conectare"))
                self.root.after(0, lambda: self.status_label.config(text="Deconectat", foreground="red"))
        
        self.async_thread.run_async(_connect())
    
    def notification_handler(self, sender, data):
        """Handler pentru notificări BLE primite de la controller"""
        # Add data to receive buffer
        self.receive_buffer.extend(data)
        
        # Process received commands
        while len(self.receive_buffer) > 0:
            if len(self.receive_buffer) < 1:
                break
                
            cmd = self.receive_buffer[0]
            
            if cmd == MOUSE_MOVE:
                expected_size = 4
            elif cmd in [KEY_PRESS, KEY_RELEASE]:
                expected_size = 3
            elif cmd == CENTER_CURSOR:
                expected_size = 4
                if len(self.receive_buffer) >= expected_size:
                    self.center_cursor()
                    self.receive_buffer = self.receive_buffer[expected_size:]
                continue
            elif cmd == GET_STATUS:
                expected_size = 4
                if len(self.receive_buffer) >= expected_size:
                    self.process_status_update()
                    self.receive_buffer = self.receive_buffer[expected_size:]
                continue
            else:
                self.receive_buffer.pop(0)
                continue
                
            if len(self.receive_buffer) < expected_size:
                break
                
            # Process the command based on its type
            if cmd == MOUSE_MOVE:
                x = struct.unpack('b', bytes([self.receive_buffer[1]]))[0]
                y = struct.unpack('b', bytes([self.receive_buffer[2]]))[0]
                self.process_mouse_move(x, y)
                
            elif cmd == KEY_PRESS:
                key_char = chr(self.receive_buffer[1])
                self.process_key_press(key_char)
                
            elif cmd == KEY_RELEASE:
                key_char = chr(self.receive_buffer[1])
                self.process_key_release(key_char)
                
            self.receive_buffer = self.receive_buffer[expected_size:]

    
    def toggle_mode(self):
        """Schimbă modul de funcționare prin trimiterea unei comenzi"""
        if self.ble_client and self.ble_client.is_connected:
            try:
                command = bytes([CHANGE_MODE])
                asyncio.run_coroutine_threadsafe(
                    self.ble_client.write_gatt_char(UART_RX_CHAR_UUID, command),
                    self.async_thread.loop
                )
                self.log("Comandă schimbare mod trimisă")
            except Exception as e:
                self.log(f"Eroare la trimiterea comenzii: {e}")
        else:
            messagebox.showwarning("Avertisment", "Nu există conexiune BLE activă!")
    
    def calibrate_sensors(self):
        """Declanșează calibrarea senzorilor"""
        if self.ble_client and self.ble_client.is_connected:
            try:
                command = bytes([CALIBRATE])
                asyncio.run_coroutine_threadsafe(
                    self.ble_client.write_gatt_char(UART_RX_CHAR_UUID, command),
                    self.async_thread.loop
                )
                self.log("Comandă calibrare trimisă")

                # Center cursor after calibration
                self.root.after(2000, self.center_cursor)

            except Exception as e:
                self.log(f"Eroare la calibrare: {e}")
        else:
            messagebox.showwarning("Avertisment", "Nu există conexiune BLE activă!")

    def request_status(self):
        """Solicită starea curentă de la ESP32"""
        if self.ble_client and self.ble_client.is_connected:
            try:
                command = bytes([GET_STATUS])
                asyncio.run_coroutine_threadsafe(
                    self.ble_client.write_gatt_char(UART_RX_CHAR_UUID, command),
                    self.async_thread.loop
                )
                self.log("Solicitare status trimisă")
            except Exception as e:
                self.log(f"Eroare la solicitarea status: {e}")
        else:
            messagebox.showwarning("Avertisment", "Nu există conexiune BLE activă!")

    def on_sensitivity_change(self, value):
        """Handler pentru schimbarea sensibilității"""
        sensitivity = int(value)
        if self.ble_client and self.ble_client.is_connected:
            try:
                command = bytes([SET_SENSITIVITY, sensitivity])
                asyncio.run_coroutine_threadsafe(
                    self.ble_client.write_gatt_char(UART_RX_CHAR_UUID, command),
                    self.async_thread.loop
                )
                self.log(f"Sensibilitate setată la: {sensitivity}")
            except Exception as e:
                self.log(f"Eroare la setarea sensibilității: {e}")

    def process_status_update(self):
        """Procesează actualizarea de status primită de la ESP32"""
        if len(self.receive_buffer) >= 4:
            mode = self.receive_buffer[1]
            mouse_sens = self.receive_buffer[2]
            keyboard_sens = self.receive_buffer[3]
            
            # Update UI elements
            self.current_mode = mode
            if mode == 0:  # MOUSE_MODE
                self.mode_button.config(text="Schimbă în Mod Keyboard")
                self.sensitivity_scale.set(mouse_sens)
                self.root.after(0, lambda: self.mode_label.config(text="Mod: Mouse"))
            else:  # KEYBOARD_MODE
                self.mode_button.config(text="Schimbă în Mod Mouse")
                self.sensitivity_scale.set(keyboard_sens)
                self.root.after(0, lambda: self.mode_label.config(text="Mod: Keyboard"))
            
            self.log(f"Status actualizat - Mod: {'Mouse' if mode == 0 else 'Keyboard'}, Sensibilitate: {mouse_sens if mode == 0 else keyboard_sens}")

    def center_cursor(self):
        """Centrează cursorul pe ecran"""
        center_x = self.screen_width // 2
        center_y = self.screen_height // 2
        self.mouse.position = (center_x, center_y)
        self.log(f"Cursor centrat la ({center_x}, {center_y})")

    def process_mouse_move(self, x, y):
        """Procesează mișcare mouse - cu decodificare corectă pentru valori cu semn"""
        # Convert bytes to signed integers
        if x > 127:
            x -= 256
        if y > 127:
            y -= 256
            
        # Filter out noise movements
        if abs(x) < 1 and abs(y) < 1:
            return
            
        # Move the mouse
        self.mouse.move(x, y)
        
        # Log only significant movements
        if abs(x) > 5 or abs(y) > 5:
            self.root.after(0, lambda: self.log(f"Mișcare mouse: ({x}, {y})"))
        
        # Update UI on mode change
        self.root.after(0, lambda: self.mode_label.config(text="Mod: Mouse"))
        
    def process_key_press(self, key_char):
        """Procesează apăsare tastă"""
        self.keyboard.press(key_char)
        self.root.after(0, lambda k=key_char: self.log(f"Tastă apăsată: {k}"))
        self.root.after(0, lambda: self.mode_label.config(text="Mod: Tastatură"))
    
    def process_key_release(self, key_char):
        """Procesează eliberare tastă"""
        self.keyboard.release(key_char)
    
    def disconnect(self):
        """Deconectare de la dispozitiv"""
        self.running = False
        self.connected = False
        self.connect_btn.config(text="Conectare")
        self.status_label.config(text="Deconectat", foreground="red")
        self.mode_label.config(text="Mod: -")
        self.log("Deconectat")
    
    def test_mouse_move(self):
        """Testează mișcarea mouse-ului"""
        self.log("Test mișcare mouse")
        for i in range(5):
            self.mouse.move(10, 0)  # Right movement
            time.sleep(0.1)
            self.mouse.move(0, 10)  # Down movement
            time.sleep(0.1)
            self.mouse.move(-10, 0)  # Left movement
            time.sleep(0.1)
            self.mouse.move(0, -10)  # Up movement
            time.sleep(0.1)
    
    def test_keyboard(self, key):
        """Testează tastatura"""
        self.log(f"Test tastă {key}")
        self.keyboard.press(key)
        time.sleep(0.2)
        self.keyboard.release(key)
    
    def on_closing(self):
        """Gestionează închiderea aplicației"""
        self.running = False
        self.async_thread.stop()
        self.root.destroy()

if __name__ == "__main__":
    # Create and run the application
    root = tk.Tk()
    app = GloveControllerBLEApp(root)
    root.protocol("WM_DELETE_WINDOW", app.on_closing)
    root.mainloop()