#include <Wire.h>
#include <LiquidCrystal_I2C.h>
#include <SD.h>
#include <SPI.h>

LiquidCrystal_I2C lcd(0x27, 16, 2);

const byte  PULSE_PIN   = A1;
const byte  RED_LED     = 6;
const byte  GREEN_LED   = 7;
const byte  BUZZER_PIN  = 5;
const byte  SD_CS       = 4;

const unsigned int CAL_TIME_MS      = 6000;
const byte         SAFETY_MARGIN    = 25;
const unsigned int MIN_BEAT_SPACING = 600;
const byte         HYSTERESIS       = 20;
const byte         SAMPLE_RATE_MS   = 10;

const unsigned long BPM_UPDATE_INTERVAL = 5000;
const unsigned long STANDBY_TIMEOUT = 3000;
unsigned long lastBpmUpdateTime = 0;

float bpmBuffer[5];
byte bpmIndex = 0;
bool bufferFull = false;

int baselineMax   = 0;
int threshold     = 0;
bool pulseHigh    = false;
unsigned long lastBeatTime = 0;
unsigned long lastPulseActivity = 0; 

#define FILTER_SIZE 5
int filterBuffer[FILTER_SIZE];
byte filterIndex = 0;

bool isInStandbyMode = false; 

void setup() {
  Serial.begin(9600);
  lcd.init();
  lcd.backlight();
  lcd.print("Calibrare...");

  pinMode(RED_LED,    OUTPUT);
  pinMode(GREEN_LED,  OUTPUT);
  pinMode(BUZZER_PIN, OUTPUT);

  if (!SD.begin(SD_CS)) {
    lcd.setCursor(0, 1);
    lcd.print("SD ERROR      ");
  } else {
    lcd.setCursor(0, 1);
    lcd.print("SD OK         ");
  }

  unsigned long t0 = millis();
  while (millis() - t0 < CAL_TIME_MS) {
    int v = analogRead(PULSE_PIN);
    if (v > baselineMax) baselineMax = v;
    delay(5);
  }
  threshold = baselineMax + SAFETY_MARGIN;

  delay(1000);
  lcd.clear();
  lcd.print("Puls: -- BPM");
  lcd.setCursor(0, 1);
  lcd.print("Astept puls...");
  
  lastPulseActivity = millis(); 
  isInStandbyMode = true;
}

void loop() {
  int raw = analogRead(PULSE_PIN);

  filterBuffer[filterIndex++] = raw;
  if (filterIndex >= FILTER_SIZE) filterIndex = 0;

  long sum = 0;
  for (byte i = 0; i < FILTER_SIZE; i++) sum += filterBuffer[i];
  int v = sum / FILTER_SIZE;

  unsigned long now = millis();

  if (v > threshold && !pulseHigh && (now - lastBeatTime) > MIN_BEAT_SPACING) {
    pulseHigh = true;
    lastPulseActivity = now; 
    isInStandbyMode = false; 

    if (lastBeatTime > 0) {
      float bpm = 60000.0 / (now - lastBeatTime);
      if (bpm >= 45 && bpm <= 160) {
        bpmBuffer[bpmIndex++] = bpm;
        Serial.println(bpm, 1);

        if (bpmIndex >= 5) {
          bpmIndex = 0;
          bufferFull = true;
        }
      }
    }

    lastBeatTime = now;
  }

  if (v < (threshold - HYSTERESIS)) {
    pulseHigh = false;
  }

  if ((bufferFull || bpmIndex >= 3) && (now - lastBpmUpdateTime >= BPM_UPDATE_INTERVAL) && !isInStandbyMode) {
    float avgBpm = 0;
    byte count = bufferFull ? 5 : bpmIndex;

    for (byte i = 0; i < count; i++) {
      avgBpm += bpmBuffer[i];
    }
    avgBpm /= count;

    afiseazaSiSemnalizeaza(avgBpm);
    logPeSD(avgBpm);
    lastBpmUpdateTime = now;
  }

  if ((now - lastPulseActivity) > STANDBY_TIMEOUT && !isInStandbyMode) {
    standbyDisplay();
    isInStandbyMode = true;
    
    bpmIndex = 0;
    bufferFull = false;
  }

  delay(SAMPLE_RATE_MS);
}

void afiseazaSiSemnalizeaza(float bpm) {
  lcd.setCursor(0, 0);
  lcd.print("Puls: ");
  lcd.print(bpm, 0);
  lcd.print(" BPM   ");

  if (bpm >= 60 && bpm <= 85) {
    digitalWrite(GREEN_LED, HIGH);
    digitalWrite(RED_LED,   LOW);
    tonBuzzerPozitiv();
    lcd.setCursor(0, 1);
    lcd.print("Puls OK!      ");
  } else {
    digitalWrite(GREEN_LED, LOW);
    digitalWrite(RED_LED,   HIGH);
    tonBuzzerNegativ();
    lcd.setCursor(0, 1);
    lcd.print("Puls anormal!");
  }
}

void standbyDisplay() {
  lcd.setCursor(0, 0);
  lcd.print("Puls: -- BPM   ");
  lcd.setCursor(0, 1);
  lcd.print("Astept puls... ");
  digitalWrite(GREEN_LED, LOW);
  digitalWrite(RED_LED,   LOW);
  noTone(BUZZER_PIN);
  
  Serial.println("Intrat in modul standby - astept puls...");
}

void logPeSD(float bpm) {
  File f = SD.open("puls.txt", FILE_WRITE);
  if (f) {
    f.print("Puls: ");
    f.print(bpm, 1);
    f.println(" BPM");
    f.close();

    Serial.print("Salvat in puls.txt - BPM: ");
    Serial.println(bpm, 1);
  } else {
    Serial.println("Eroare la scrierea in SD card!");
  }
}