/*
 * sparrow_web_bme - ESP32-C6 + BME680 + Wi-Fi + tiny HTTP server
 *
 * - Scans Wi-Fi, connects to CONFIG_WIFI_SSID/CONFIG_WIFI_PASSWORD
 * - Starts DHCPv4 once
 * - Prints IPv4 address when acquired
 * - Serves latest BME680 sample as JSON at http://<ip>:8080/
 *
 * Notes:
 * - Uses zsock_* APIs (no undefined references to socket/bind/etc)
 * - Does NOT redefine sensor_value_to_float (Zephyr already provides it)
 * - net_mgmt event id type is uint64_t in newer Zephyr
 */

//  west build -b esp32c6_devkitc/esp32c6/hpcore -p auto . 
//  west flash -d build --runner esp32  
//  west espressif monitor

#include <zephyr/kernel.h>
#include <zephyr/logging/log.h>
#include <zephyr/drivers/sensor.h>

#include <zephyr/net/net_if.h>
#include <zephyr/net/net_mgmt.h>
#include <zephyr/net/net_event.h>
#include <zephyr/net/wifi_mgmt.h>
#include <zephyr/net/dhcpv4.h>
#include <zephyr/net/socket.h>
#include <zephyr/net/net_ip.h>

#include <zephyr/sys/atomic.h>
#include <string.h>
#include <stdio.h>

LOG_MODULE_REGISTER(sparrow_web_bme, LOG_LEVEL_INF);

/* ====== Config ====== */
#define HTTP_PORT        8080 /* HTTP server port */
#define HTTP_BACKLOG     1    /* Pending connections */
#define SENSOR_PERIOD_MS 2000 /* BME680 poll interval */

/* ====== Data types ====== */
struct bme_sample {
	uint32_t seq;   /* Sample sequence counter */
	float t_c;      /* Temperature in Celsius */
	float rh;       /* Relative humidity percent */
	float p_kpa;    /* Pressure in kPa */
	float gas_ohm;  /* Gas resistance in ohms */
};

/* ====== Shared state ====== */
static struct bme_sample g_sample;     /* Latest sample served over HTTP */
static struct k_mutex g_sample_lock;   /* Protects g_sample */

static atomic_t g_wifi_connected;      /* Associated to AP */
static atomic_t g_dhcp_started;        /* DHCP started once */
static atomic_t g_have_ipv4;           /* IPv4 address acquired */
static atomic_t g_http_started;        /* Log HTTP start once */

/* net_mgmt callbacks (must not mix layer codes in one mask) */
static struct net_mgmt_event_callback g_mgmt_wifi_cb; /* Wi-Fi events */
static struct net_mgmt_event_callback g_mgmt_ipv4_cb; /* IPv4 events */

/* ====== Forward declarations ====== */
static const struct device *get_bme680_device(void);
static void sensor_poll_thread(void *a, void *b, void *c);
static void http_server_thread(void *a, void *b, void *c);
static void start_wifi_scan(struct net_if *iface);
static void wifi_connect(struct net_if *iface);
static void print_ipv4_addr(struct net_if *iface);
static void mgmt_event_handler(struct net_mgmt_event_callback *cb,
			       uint64_t mgmt_event,
			       struct net_if *iface);

/* ====== Wi-Fi + DHCP + IP printing ====== */

/* Start a Wi-Fi scan on the interface. */
static void start_wifi_scan(struct net_if *iface)
{
	int rc = net_mgmt(NET_REQUEST_WIFI_SCAN, iface, NULL, 0);
	if (rc) {
		LOG_ERR("[wifi] scan start failed: %d", rc);
	} else {
		LOG_INF("[wifi] scanning...");
	}
}

/* Request Wi-Fi connect with configured credentials. */
static void wifi_connect(struct net_if *iface)
{
	struct wifi_connect_req_params cnx = {0};

	cnx.ssid = (const uint8_t *)CONFIG_WIFI_SSID;
	cnx.ssid_length = strlen(CONFIG_WIFI_SSID);

	cnx.psk = (const uint8_t *)CONFIG_WIFI_PASSWORD;
	cnx.psk_length = strlen(CONFIG_WIFI_PASSWORD);

	cnx.security = WIFI_SECURITY_TYPE_PSK;
	cnx.channel = WIFI_CHANNEL_ANY;
	cnx.band = WIFI_FREQ_BAND_2_4_GHZ; /* safe default for typical APs */

	LOG_INF("[wifi] connecting to '%s' ...", CONFIG_WIFI_SSID);

	int rc = net_mgmt(NET_REQUEST_WIFI_CONNECT, iface, &cnx, sizeof(cnx));
	if (rc) {
		LOG_ERR("[wifi] connect request failed: %d", rc);
	}
}

/* Log the first usable IPv4 address on the interface. */
static void print_ipv4_addr(struct net_if *iface)
{
	/* Use net_if_config_ipv4_get() in Zephyr 4.x (signature: (iface, &ipv4)) */
	struct net_if_ipv4 *ipv4 = NULL;
	int rc = net_if_config_ipv4_get(iface, &ipv4);
	if (rc || !ipv4) {
		LOG_WRN("[ip] IPv4 config not available yet (rc=%d)", rc);
		return;
	}

	/* Best-effort: look for the first valid IPv4 unicast address */
	for (int i = 0; i < NET_IF_MAX_IPV4_ADDR; i++) {
		struct net_if_addr *ifaddr = &ipv4->unicast[i];
		/* net_if_addr is a common type; check family */
		if (ifaddr->is_used && ifaddr->addr_type != NET_ADDR_ANY &&
		    ifaddr->address.family == AF_INET) {
			char ip[NET_IPV4_ADDR_LEN];
			(void)net_addr_ntop(AF_INET, &ifaddr->address.in_addr, ip, sizeof(ip));
			LOG_INF("[ip] %s", ip);
			return;
		}
	}

	LOG_WRN("[ip] no IPv4 address found in unicast list yet");
}

/* Handle Wi-Fi and IPv4 net_mgmt events. */
static void mgmt_event_handler(struct net_mgmt_event_callback *cb,
			       uint64_t mgmt_event,
			       struct net_if *iface)
{
	ARG_UNUSED(cb);

	switch (mgmt_event) {
	case NET_EVENT_WIFI_SCAN_RESULT: {
		const struct wifi_scan_result *r = (const struct wifi_scan_result *)cb->info;
		if (r && r->ssid_length > 0) {
			LOG_INF("[scan] ssid='%.*s' chan=%u rssi=%d sec=%u",
				r->ssid_length, r->ssid,
				r->channel, r->rssi, r->security);
		}
		break;
	}

	case NET_EVENT_WIFI_SCAN_DONE:
		LOG_INF("[scan] done");
		/* Connect after scan completes (simple flow) */
		wifi_connect(iface);
		break;

	case NET_EVENT_WIFI_CONNECT_RESULT: {
		const struct wifi_status *st = (const struct wifi_status *)cb->info;
		int status = st ? st->status : -1;

		LOG_INF("[wifi] CONNECT_RESULT: status=%d", status);

		if (status == 0) {
			atomic_set(&g_wifi_connected, 1);

			/* DHCP start only once */
			if (!atomic_cas(&g_dhcp_started, 0, 1)) {
				break;
			}

			LOG_INF("[dhcp] starting DHCPv4");
			net_dhcpv4_start(iface);
		}
		break;
	}

	case NET_EVENT_WIFI_DISCONNECT_RESULT:
		LOG_WRN("[wifi] disconnected");
		atomic_set(&g_wifi_connected, 0);
		atomic_set(&g_have_ipv4, 0);
		atomic_set(&g_dhcp_started, 0);
		break;

	case NET_EVENT_IPV4_ADDR_ADD:
		LOG_INF("[ip] IPv4 addr added");
		atomic_set(&g_have_ipv4, 1);
		print_ipv4_addr(iface);

		if (!atomic_cas(&g_http_started, 0, 1)) {
			break;
		}
		LOG_INF("[http] starting server on port %d", HTTP_PORT);
		break;

	default:
		break;
	}
}

/* ====== BME680 sampling ====== */

/* Return the ready BME680 device, or NULL if unavailable. */
static const struct device *get_bme680_device(void)
{
	/* This works with Zephyr's Bosch BME680 driver instances */
	const struct device *dev = DEVICE_DT_GET_ANY(bosch_bme680);
	if (dev && device_is_ready(dev)) {
		return dev;
	}
	return NULL;
}

/* Periodically read the sensor and update g_sample. */
static void sensor_poll_thread(void *a, void *b, void *c)
{
	ARG_UNUSED(a);
	ARG_UNUSED(b);
	ARG_UNUSED(c);

	const struct device *bme = get_bme680_device();
	if (!bme) {
		LOG_ERR("BME680 device not found/ready (check overlay + wiring + I2C addr).");
		while (1) {
			k_sleep(K_SECONDS(2));
		}
	}

	LOG_INF("BME680 ready.");

	uint32_t seq = 0;

	while (1) {
		int rc = sensor_sample_fetch(bme);
		if (rc) {
			LOG_WRN("sensor_sample_fetch failed: %d", rc);
			k_sleep(K_MSEC(SENSOR_PERIOD_MS));
			continue;
		}

		struct sensor_value t, rh, p, gas;

		rc = sensor_channel_get(bme, SENSOR_CHAN_AMBIENT_TEMP, &t);
		rc |= sensor_channel_get(bme, SENSOR_CHAN_HUMIDITY, &rh);
		rc |= sensor_channel_get(bme, SENSOR_CHAN_PRESS, &p);
		rc |= sensor_channel_get(bme, SENSOR_CHAN_GAS_RES, &gas);

		if (rc) {
			LOG_WRN("sensor_channel_get failed: %d", rc);
			k_sleep(K_MSEC(SENSOR_PERIOD_MS));
			continue;
		}

		const double t_c = sensor_value_to_double(&t);
		const double rh_pc = sensor_value_to_double(&rh);
		const double p_kpa = sensor_value_to_double(&p) / 1000.0; /* Pa -> kPa (typical) */
		const double gas_ohm = sensor_value_to_double(&gas);

		seq++;

		k_mutex_lock(&g_sample_lock, K_FOREVER);
		g_sample.seq = seq;
		g_sample.t_c = (float)t_c;
		g_sample.rh = (float)rh_pc;
		g_sample.p_kpa = (float)p_kpa;
		g_sample.gas_ohm = (float)gas_ohm;
		k_mutex_unlock(&g_sample_lock);

		LOG_INF("#%u T=%.2fC RH=%.2f%% P=%.2fkPa GAS=%.0fΩ",
			seq, (float)t_c, (float)rh_pc, (float)p_kpa, (float)gas_ohm);

		k_sleep(K_MSEC(SENSOR_PERIOD_MS));
	}
}

/* BME680 polling thread. */
K_THREAD_DEFINE(sensor_tid, 3072, sensor_poll_thread, NULL, NULL, NULL, 5, 0, 0);

/* ====== HTTP Server ====== */

/* Serve the latest sample as JSON over a minimal HTTP endpoint. */
static void http_server_thread(void *a, void *b, void *c)
{
	ARG_UNUSED(a);
	ARG_UNUSED(b);
	ARG_UNUSED(c);

	/* Wait until we have IPv4 before starting */
	while (!atomic_get(&g_have_ipv4)) {
		k_sleep(K_MSEC(200));
	}

	int s = zsock_socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
	if (s < 0) {
		LOG_ERR("[http] socket() failed: %d", errno);
		return;
	}

	struct sockaddr_in addr;
	memset(&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_port = htons(HTTP_PORT);
	addr.sin_addr.s_addr = htonl(INADDR_ANY);

	if (zsock_bind(s, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
		LOG_ERR("[http] bind() failed: %d", errno);
		zsock_close(s);
		return;
	}

	if (zsock_listen(s, HTTP_BACKLOG) < 0) {
		LOG_ERR("[http] listen() failed: %d", errno);
		zsock_close(s);
		return;
	}

	LOG_INF("[http] ready on port %d (GET http://<ip>:%d/ )", HTTP_PORT, HTTP_PORT);

	while (1) {
		struct sockaddr_in client;
		socklen_t clen = sizeof(client);
		int cfd = zsock_accept(s, (struct sockaddr *)&client, &clen);
		if (cfd < 0) {
			/* Keep running even if accept fails transiently */
			k_sleep(K_MSEC(50));
			continue;
		}

		char cip[NET_IPV4_ADDR_LEN];
		(void)net_addr_ntop(AF_INET, &client.sin_addr, cip, sizeof(cip));
		LOG_INF("[http] client connected: %s", cip);

		/* Read and ignore request (small buffer) */
		char req[256];
		(void)zsock_recv(cfd, req, sizeof(req) - 1, 0);

		/* Copy latest sample */
		struct bme_sample s0;
		k_mutex_lock(&g_sample_lock, K_FOREVER);
		s0 = g_sample;
		k_mutex_unlock(&g_sample_lock);

		/* Build response body (JSON) */
		char body[256];
		int blen = snprintk(body, sizeof(body),
				   "{"
				   "\"seq\":%u,"
				   "\"t_c\":%.2f,"
				   "\"rh\":%.2f,"
				   "\"p_kpa\":%.2f,"
				   "\"gas_ohm\":%.0f"
				   "}\n",
				   s0.seq, s0.t_c, s0.rh, s0.p_kpa, s0.gas_ohm);
		if (blen < 0) {
			blen = 0;
		}

		/* Headers */
		char hdr[256];
		int hlen = snprintk(hdr, sizeof(hdr),
				    "HTTP/1.1 200 OK\r\n"
				    "Content-Type: application/json\r\n"
				    "Connection: close\r\n"
				    "Content-Length: %d\r\n"
				    "\r\n",
				    blen);
		if (hlen > 0) {
			(void)zsock_send(cfd, hdr, hlen, 0);
		}
		if (blen > 0) {
			(void)zsock_send(cfd, body, blen, 0);
		}

		zsock_close(cfd);
	}
}

/* HTTP server thread. */
K_THREAD_DEFINE(http_tid, 4096, http_server_thread, NULL, NULL, NULL, 6, 0, 0);

/* ====== main ====== */

/* Initialize shared state and start Wi-Fi + DHCP flow. */
void main(void)
{
	k_mutex_init(&g_sample_lock);

	LOG_INF("Booted.");

	struct net_if *iface = net_if_get_default();
	if (!iface) {
		LOG_ERR("No default network interface");
		return;
	}

	/* Listen for Wi-Fi + IPv4 events */
	net_mgmt_init_event_callback(
		&g_mgmt_wifi_cb,
		mgmt_event_handler,
		NET_EVENT_WIFI_SCAN_RESULT |
		NET_EVENT_WIFI_SCAN_DONE |
		NET_EVENT_WIFI_CONNECT_RESULT |
		NET_EVENT_WIFI_DISCONNECT_RESULT);

	net_mgmt_add_event_callback(&g_mgmt_wifi_cb);

	net_mgmt_init_event_callback(
		&g_mgmt_ipv4_cb,
		mgmt_event_handler,
		NET_EVENT_IPV4_ADDR_ADD);

	net_mgmt_add_event_callback(&g_mgmt_ipv4_cb);

	/* Kick off flow */
	start_wifi_scan(iface);
}
